<?php

namespace Ignite\Theme\Console;

use Illuminate\Console\Command;
use RuntimeException;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Process\Process;

/**
 * Builds theme assets using Node.js.
 *
 * This command checks and installs the required Node.js version using NVM,
 * installs theme dependencies, and compiles assets for either development or production.
 */
class ThemeBuildCommand extends Command
{
    /**
     * The default Node.js version to use if not specified in package.json or options.
     */
    const DEFAULT_NODE_VERSION = '22.16.0';

    /**
     * @var string
     */
    protected $name = 'ignite:theme:build';

    /**
     * @var string
     */
    protected $description = 'Build theme assets using Node.js. Checks and installs required Node.js version,'
        . ' installs dependencies, and compiles assets.';

    /**
     * @var string|null
     */
    protected $nodeVersion;

    /**
     * Get the console command arguments.
     *
     * This method defines the arguments accepted by the command.
     *
     * @return array An array of argument definitions.
     */
    public function getArguments()
    {
        return [];
    }

    /**
     * Get the console command options.
     *
     * This method defines the options accepted by the command, including theme, engine, node version, and dev flag.
     *
     * @return array An array of option definitions.
     */
    public function getOptions()
    {
        // @phpcs:disable Generic.Files.LineLength
        return [
            ['theme', 't', InputOption::VALUE_REQUIRED, 'Name of the theme to process. It will guess from themes folder if not provided.', null],
            ['engine', 'e', InputOption::VALUE_REQUIRED, 'Choose either npm or yarn.', 'yarn'],
            ['node_version', null, InputOption::VALUE_REQUIRED, 'Which node version to use. Uses the version from package.json if available, otherwise defaults to ' . self::DEFAULT_NODE_VERSION . '.', null],
            ['dev', 'd', InputOption::VALUE_NONE, 'If true, it runs the dev command instead of production.', null],
            ['logo', null, InputOption::VALUE_REQUIRED, 'Path to the logo file to be moved to theme/THEMENAME/images folder. It should be a PNG format.', null],
            ['banner', null, InputOption::VALUE_REQUIRED, 'Path to the banner file to be moved to theme/THEMENAME/images folder. It should be a JPG format.', null],
            ['email', null, InputOption::VALUE_REQUIRED, 'Path to the email file to be moved to theme/THEMENAME/images folder. It should be a JPG format.', null],
            ['primary-color', null, InputOption::VALUE_REQUIRED, 'Primary color for the theme in hex format (e.g. #FF0000).', null],
            ['secondary-color', null, InputOption::VALUE_REQUIRED, 'Secondary color for the theme in hex format (e.g. #00FF00).', null],
        ];
        // @phpcs:enable
    }

    /**
     * This method orchestrates the theme build process.
     *
     * @return void
     */
    public function handle()
    {
        // Determine the theme to build, either from the command option or by guessing.
        $theme = $this->option('theme') ?: $this->guessTheme();
        if (!$theme) {
            throw new RuntimeException('No theme specified and no themes found.');
        }

        try {
            // Handle logo and banner files if provided
            $this->handleImageFiles($theme);
        } catch (\Exception $e) {
            $this->error("Error handling theme images: " . $e->getMessage());
        }

        try {
            // Handle color options if provided
            $this->handleColorOptions($theme);
        } catch (\Exception $e) {
            $this->error("Error handling theme colors: " . $e->getMessage());
        }

        // Set the package manager (npm or yarn) and the required Node.js version.
        $engine = $this->option('engine') ?: 'yarn';
        $this->nodeVersion = $this->option('node_version') ?: $this->guessNodeVersion($theme);
        $isDev = $this->option('dev') ? true : false;

        // Verify the theme directory exists.
        $themePath = base_path("themes/{$theme}");
        if (!is_dir($themePath)) {
            throw new RuntimeException("Theme directory not found: {$themePath}");
        }

        // Check the current Node.js version and install the required version if necessary using NVM.
        $currentVersion = $this->getCurrentNodeVersion();
        if (empty($currentVersion) || version_compare($currentVersion, $this->nodeVersion, '<')) {
            $this->info("Current Node.js version ({$currentVersion}) is less than required version ({$this->nodeVersion}).");
            if (!$this->installNodeVersion()) {
                return;
            }
            $currentVersion = $this->getCurrentNodeVersion();
            if (empty($currentVersion) || version_compare($currentVersion, $this->nodeVersion, '<')) {
                throw new RuntimeException("Node.js version {$this->nodeVersion} could not be installed or is still not active.");
            }
        }
        $this->info("Node.js version installed: {$currentVersion}");

        // Run the package installation command (`npm install` or `yarn install`) in the theme directory.
        $this->info("Running {$engine} install...");
        if (!$this->runPackageInstall($themePath, $engine)) {
            throw new RuntimeException("Failed to run {$engine} install");
        }

        // Execute the appropriate build command (`dev` or `production`) using the chosen package manager.
        $command = $isDev ? 'dev' : 'production';
        $this->info("Running {$engine} run {$command}...");
        if (!$this->runBuildCommand($themePath, $engine, $command)) {
            throw new RuntimeException("Failed to run {$engine} run {$command}");
        }

        $this->info('<fg=green>✓</> Theme build completed successfully.');
    }

    /**
     * Convert JPEG image to PNG format if the input file is a JPEG.
     *
     * This method checks if the input image file is a JPEG (either .jpg or .jpeg extension)
     * and converts it to PNG format if necessary. The conversion is done using PHP's GD library.
     * The converted PNG file is saved to a temporary location and the path to the new file is returned.
     *
     * @param string $imgFile Path to the input image file
     * @return string Path to the PNG file (either the converted file or the original if no conversion was needed)
     * @throws RuntimeException If the JPEG file cannot be read or if the PNG conversion fails
     */
    protected function convertJpegToPngIfNeeded(string $imgFile): string
    {
        // Convert JPEG to PNG if needed
        if (strtolower(pathinfo($imgFile, PATHINFO_EXTENSION)) === 'jpg'
            || strtolower(pathinfo($imgFile, PATHINFO_EXTENSION)) === 'jpeg'
        ) {
            $image = imagecreatefromjpeg($imgFile);
            if ($image === false) {
                throw new RuntimeException("Failed to create image from JPEG: {$imgFile}");
            }

            // Create a temporary PNG file
            $tempPngPath = tempnam(sys_get_temp_dir(), 'logo_') . '.png';
            if (!imagepng($image, $tempPngPath)) {
                imagedestroy($image);
                throw new RuntimeException("Failed to convert JPEG to PNG: {$imgFile}");
            }

            imagedestroy($image);
            $imgFile = $tempPngPath;
        }

        return $imgFile;
    }

    /**
     * Convert PNG image to JPEG format if the input file is a PNG.
     *
     * This method checks if the input image file is a PNG and converts it to JPEG format if necessary.
     * The conversion is done using PHP's GD library. The converted JPEG file is saved to a temporary
     * location and the path to the new file is returned.
     *
     * @param string $imgFile Path to the input image file
     * @return string Path to the JPEG file (either the converted file or the original if no conversion was needed)
     * @throws RuntimeException If the PNG file cannot be read or if the JPEG conversion fails
     */
    protected function convertPngToJpgIfNeeded(string $imgFile): string
    {
        // Convert PNG to JPEG if needed
        if (strtolower(pathinfo($imgFile, PATHINFO_EXTENSION)) === 'png') {
            $image = imagecreatefrompng($imgFile);
            if ($image === false) {
                throw new RuntimeException("Failed to create image from PNG: {$imgFile}");
            }

            // Create a temporary JPEG file
            $tempJpgPath = tempnam(sys_get_temp_dir(), 'banner_') . '.jpg';
            // 95 is the quality (0-100)
            if (!imagejpeg($image, $tempJpgPath, 95)) {
                imagedestroy($image);
                throw new RuntimeException("Failed to convert PNG to JPEG: {$imgFile}");
            }

            imagedestroy($image);
            $imgFile = $tempJpgPath;
        }

        return $imgFile;
    }

    /**
     * Execute a command with NVM loaded.
     *
     * This method prepares the shell environment by loading NVM and optionally using a specific Node.js version
     * before executing the given command. It handles changing the working directory if a path is provided.
     *
     * @param string $command The command to execute.
     * @param string|null $path The working directory (optional). If provided, the command will be executed in this directory.
     * @param bool $useNvm Whether to run `nvm use` with the current version before executing the command.
     * @param callable|null $callback Optional callback for process output. It receives the output type and buffer as arguments.
     * @return bool True if the command executed successfully, false otherwise.
     */
    protected function executeWithNvm(string $command, ?string $path = null, bool $useNvm = false, ?callable $callback = null): bool
    {
        try {
            $nvmLoad = 'export NVM_DIR="/home/$(whoami)/.nvm" && [ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh"';
            if ($useNvm && $this->nodeVersion) {
                $nvmLoad .= " && nvm use {$this->nodeVersion}";
            }

            $fullCmd = !empty($path)
                ? "{$nvmLoad} && cd {$path} && {$command}"
                : "{$nvmLoad} && {$command}";

            $process = Process::fromShellCommandline($fullCmd);
            $process->setTimeout(300); // 5 minutes timeout

            if ($callback) {
                $process->run($callback);
            } else {
                // $process->run(function ($type, $buffer) { echo '--- ' . $buffer; });
                $process->run();
            }

            return $process->isSuccessful();
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Get the current Node.js version using NVM.
     *
     * This method executes `nvm current` and parses the output to return the version number.
     *
     * @return string|null The current Node.js version string, or null if not found or command fails.
     */
    protected function getCurrentNodeVersion(): ?string
    {
        $output = '';
        $success = $this->executeWithNvm(
            'nvm current',
            useNvm: true,
            callback: function ($type, $buffer) use (&$output) {
                $output .= $buffer;
            }
        );

        if ($success) {
            $output = trim($output);
            // nvm current outputs something like "v14.6.0 (-> v14.6.0)"
            if (preg_match('/v(\d+\.\d+\.\d+)/', $output, $matches)) {
                return $matches[1];
            }
        }

        return null;
    }

    /**
     * Get the Theme manager instance.
     *
     * @return \Ignite\Theme\Manager The theme manager instance.
     */
    protected function getThemeManager()
    {
        return $this->laravel['theme'];
    }

    /**
     * Guesses the required Node.js version for the theme.
     *
     * It first checks the theme's package.json for an "engines.node" field.
     * If found, it extracts and returns the version. Otherwise, it defaults to `self::DEFAULT_NODE_VERSION`.
     *
     * @param string $theme The name of the theme.
     * @return string The guessed or default Node.js version.
     */
    protected function guessNodeVersion($theme): string
    {
        $path = base_path("themes/{$theme}/package.json");
        if (!file_exists($path)) {
            return self::DEFAULT_NODE_VERSION;
        }

        $packageJson = json_decode(file_get_contents($path), true);
        if (isset($packageJson['engines']['node'])) {
            $nodeVersion = $packageJson['engines']['node'];
            // remove the version constraint
            $nodeVersion = preg_replace('/[^\d.]/', '', $nodeVersion);
            return $nodeVersion;
        }

        return self::DEFAULT_NODE_VERSION;
    }

    /**
     * Guesses the theme name based on available themes.
     *
     * If there is only one theme, it returns that theme's name. If there are multiple themes,
     * it returns the name of the first theme not called 'default'. If no themes are found
     * or all themes are 'default', it returns an empty string.
     *
     * @return string The guessed theme name, or an empty string.
     */
    protected function guessTheme(): string
    {
        $themes = $this->getThemeManager()->themes();

        // If there's only one theme, use that one
        if (count($themes) === 1) {
            return $themes[0]->name();
        }

        // If there are multiple themes, use the first one not called 'default'
        foreach ($themes as $theme) {
            if ($theme->name() !== 'default') {
                return $theme->name();
            }
        }

        // If all themes are 'default' or no themes found, return empty string
        return '';
    }

    /**
     * Handle primary and secondary color options.
     *
     * @param string $theme The name of the theme.
     * @return void
     */
    protected function handleColorOptions(string $theme): void
    {
        $themePath = base_path("themes/{$theme}");
        $scssPath = "{$themePath}/src/scss/_variables.scss";

        // Only proceed if we have color options and the SCSS file exists
        if (!file_exists($scssPath)) {
            throw new RuntimeException("SCSS variables file not found: {$scssPath}");
        }

        $scssContent = file_get_contents($scssPath);

        // Handle primary color
        if ($primaryColor = $this->option('primary-color')) {
            if (!preg_match('/^#[0-9A-Fa-f]{6}$/', $primaryColor)) {
                throw new RuntimeException("Invalid primary color format. Please use hex format (e.g. #FF0000)");
            }
            $scssContent = preg_replace(
                '/\$brand-primary:\s*#[0-9A-Fa-f]{6};/',
                '$brand-primary: ' . $primaryColor . ';',
                $scssContent,
                1,
                $count
            );
            $this->info("Primary color set to {$primaryColor} to file {$scssPath}");
        }

        // Handle secondary color
        if ($secondaryColor = $this->option('secondary-color')) {
            if (!preg_match('/^#[0-9A-Fa-f]{6}$/', $secondaryColor)) {
                throw new RuntimeException("Invalid secondary color format. Please use hex format (e.g. #FF0000)");
            }
            $scssContent = preg_replace(
                '/\$brand-secondary:\s*#[0-9A-Fa-f]{6};/',
                '$brand-secondary: ' . $secondaryColor . ';',
                $scssContent,
                1,
                $count
            );
            $this->info("Secondary color set to {$secondaryColor} to file {$scssPath}");
        }

        file_put_contents($scssPath, $scssContent);
    }

    /**
     * Handle moving logo and banner files to the theme's images folder.
     *
     * @param string $theme The name of the theme.
     * @return void
     */
    protected function handleImageFiles(string $theme): void
    {
        $imagesDir = base_path("themes/{$theme}/dist/images");

        // Create images directory if it doesn't exist
        if (!is_dir($imagesDir)) {
            mkdir($imagesDir, 0755, true);
        }

        // Handle logo file
        if ($logoPath = $this->option('logo')) {
            if (!file_exists($logoPath)) {
                throw new RuntimeException("Logo file not found: {$logoPath}");
            }
            $logoPath = $this->convertJpegToPngIfNeeded($logoPath);
            $targetLogoPath = "{$imagesDir}/logo.png";
            if (!copy($logoPath, $targetLogoPath)) {
                throw new RuntimeException("Failed to copy logo file to: {$targetLogoPath}");
            }
            $this->info("Logo file copied to: {$targetLogoPath}");
        }

        // Handle banner file
        if ($bannerPath = $this->option('banner')) {
            if (!file_exists($bannerPath)) {
                throw new RuntimeException("Banner file not found: {$bannerPath}");
            }
            $bannerPath = $this->convertPngToJpgIfNeeded($bannerPath);
            $targetBannerPath = "{$imagesDir}/banner.jpg";
            if (!copy($bannerPath, $targetBannerPath)) {
                throw new RuntimeException("Failed to copy banner file to: {$targetBannerPath}");
            }
            $this->info("Banner file copied to: {$targetBannerPath}");
        }

        // Handle email file
        if ($emailPath = $this->option('email')) {
            if (!file_exists($emailPath)) {
                throw new RuntimeException("Email file not found: {$emailPath}");
            }
            $emailPath = $this->convertPngToJpgIfNeeded($emailPath);
            $targetEmailPath = "{$imagesDir}/email_banner.jpg";
            if (!copy($emailPath, $targetEmailPath)) {
                throw new RuntimeException("Failed to copy email file to: {$targetEmailPath}");
            }
            $this->info("Email file copied to: {$targetEmailPath}");
        }
    }

    /**
     * Install a specific Node.js version using NVM.
     *
     * This method checks if NVM is installed and installs it if not. Then, it uses NVM
     * to install the Node.js version specified by `$this->nodeVersion`.
     *
     * @return bool True if the installation was successful, false otherwise.
     */
    protected function installNodeVersion(): bool
    {
        $this->info("Installing Node.js version {$this->nodeVersion}...");

        try {
            if (!$this->executeWithNvm('command -v nvm', useNvm: false)) {
                // Install NVM if not present
                $this->info('Installing NVM first...');
                $installCmd = 'curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.40.3/install.sh | bash';
                $this->line(">> RUNNING: {$installCmd}");
                $process = Process::fromShellCommandline($installCmd);
                $process->run();

                if (!$process->isSuccessful()) {
                    throw new RuntimeException('Failed to install NVM: ' . $process->getOutput());
                }
            }

            // Install and use the specified Node.js version
            $nvmCmd = "nvm install {$this->nodeVersion}";
            $this->line(">> RUNNING: {$nvmCmd}");
            return $this->executeWithNvm($nvmCmd, useNvm: false);
        } catch (\Exception $e) {
            throw new RuntimeException("Error installing Node.js: " . $e->getMessage());
        }

        return false;
    }

    /**
     * Run build command using npm or yarn.
     *
     * This method executes the specified build command (dev or production) using the chosen package manager (npm or yarn)
     * within the theme directory, ensuring the correct Node.js version is used via NVM.
     *
     * @param string $path The path to the theme directory.
     * @param string $engine The package manager to use ('npm' or 'yarn').
     * @param string $command The build command to run ('dev' or 'production').
     * @return bool True if the build command was successful, false otherwise.
     */
    protected function runBuildCommand(string $path, string $engine, string $command): bool
    {
        return $this->executeWithNvm(
            "{$engine} run {$command}",
            path: $path,
            useNvm: true,
            callback: function ($type, $buffer) {
                if ($type === Process::OUT) {
                    echo '> ' . $buffer;
                } elseif ($type === Process::ERR) {
                    echo "> ERROR: " . $buffer;
                }
            }
        );
    }

    /**
     * Run package installation using npm or yarn.
     *
     * This method executes the package installation command (`npm install` or `yarn install`)
     * within the theme directory, ensuring the correct Node.js version is used via NVM.
     *
     * @param string $path The path to the theme directory.
     * @param string $engine The package manager to use ('npm' or 'yarn').
     * @return bool True if the package installation was successful, false otherwise.
     */
    protected function runPackageInstall(string $path, string $engine): bool
    {
        $installCmd = $engine === 'yarn' ? 'yarn install' : 'npm install';
        return $this->executeWithNvm(
            $installCmd,
            path: $path,
            useNvm: true,
            callback: function ($type, $buffer) {
                if ($type === Process::OUT) {
                    echo '> ' . $buffer;
                } elseif ($type === Process::ERR) {
                    echo "> ERROR: " . $buffer;
                }
            }
        );
    }
}
