<?php

namespace Ignite\Theme\Console;

use Illuminate\Console\Command;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputArgument;

class ScaffoldCommand extends Command
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'ignite:theme:scaffold';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Scaffold a new theme.';

    /**
     * The base themes folder.
     *
     * @var string
     */
    protected $folder = 'themes';

    /**
     * The default folder structure for a theme.
     *
     * @var array
     */
    protected $structure = [
        'dist',
        'src',
        'views'
    ];

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $name = $this->input->getArgument('name');
        $parent = $this->input->getOption('parent');
        $path = $this->input->getOption('path');
        $description = $this->input->getOption('description');

        if ($this->getFilesystemManager()->exists($path = $this->determinePath($name, $path))) {
            $this->output->writeln("<fg=green>✓</> Theme `{$name}` already exists.");
            return;
        }

        $this->getEventDispatcher()->dispatch('ignite.theme.scaffolding', [
            'input' => $this->input,
            'output' => $this->output,
            'path' => $path
        ]);

        $this->createThemeDirectory($path)
             ->createThemeJson($path, $name, $parent, $description)
             ->createDirectories($path);

        $this->getEventDispatcher()->dispatch('ignite.theme.scaffolded', [
            'input' => $this->input,
            'output' => $this->output,
            'path' => $path
        ]);

        $this->info("<fg=green>✓</> Theme `{$name}` scaffold complete.");
    }

    /**
     * Create the theme directory.
     *
     * @param  string $path
     * @return self
     */
    protected function createThemeDirectory($path)
    {
        $this->getFilesystemManager()->makeDirectory($path, 0755, true);

        return $this;
    }

    /**
     * Create the theme JSON file.
     *
     * @param  string      $path
     * @param  string      $name
     * @param  string|null $parent
     * @param  string|null $description
     * @return self
     */
    protected function createThemeJson($path, $name, $parent = null, $description = null)
    {
        $this->info('Creating the theme.json file...');

        $destination = $path . DIRECTORY_SEPARATOR . 'theme.json';
        $contents = $this->generateJson(compact('name', 'parent', 'description'));

        file_put_contents($destination, $contents);

        return $this;
    }

    /**
     * Create the directories we need.
     *
     * @param  string $path
     * @return self
     */
    protected function createDirectories($path)
    {
        $this->output->writeln('Creating the default directories...');

        foreach ($this->structure as $folder) {
            $this->createThemeDirectory($path . DIRECTORY_SEPARATOR . $folder);
        }

        return $this;
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [
            ['name', InputArgument::REQUIRED, "The name of the theme."],
        ];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        return [
            ['path', null, InputOption::VALUE_OPTIONAL, "The path to the directory the theme should be installed.", $this->folder],
            ['parent', null, InputOption::VALUE_OPTIONAL, "The parent theme to inherit views and assets.", null],
            ['description', null, InputOption::VALUE_OPTIONAL, "A short description for the theme.", null],
        ];
    }

    /**
     * Determine the correct path to generate the theme.
     *
     * @param  string $name
     * @param  string $path
     * @return string
     * @throws \Exception
     */
    protected function determinePath($name, $path)
    {
        if ($path === $this->folder) {
            $path = rtrim(base_path($path), DIRECTORY_SEPARATOR);
        } else {
            $path = $path . DIRECTORY_SEPARATOR . $this->folder;
        }

        return $this->normalizePath($path . DIRECTORY_SEPARATOR . $name);
    }

    /**
     * Generate the JSON file given the provided data.
     *
     * @param  array $data
     * @return string
     */
    protected function generateJson($data)
    {
        return json_encode(array_filter($data), JSON_PRETTY_PRINT);
    }

    /**
     * Normalize the path for use across different OSs.
     *
     * @param  string $path
     * @return string
     */
    protected function normalizePath($path)
    {
        return str_replace('/', DIRECTORY_SEPARATOR, $path);
    }

    /**
     * The Filesystem manager class.
     *
     * @return \Illuminate\Filesystem\Filesystem
     */
    protected function getFilesystemManager()
    {
        return $this->laravel['files'];
    }

    /**
     * The event dispatcher.
     *
     * @return \Illuminate\Events\Dispatcher
     */
    protected function getEventDispatcher()
    {
        return $this->laravel['events'];
    }
}
