<?php

namespace Ignite\Theme\Tests;

use Ignite\Theme\Exception\ThemeNotFoundException;
use Ignite\Theme\Factory;
use Ignite\Theme\Manager;
use Ignite\Theme\Theme;
use Illuminate\Support\Arr;

class ManagerTest extends TestCase
{
    /** @var Manager */
    protected $manager;

    /** @var Factory */
    protected $factory;

    public function setUp(): void
    {
        parent::setUp();

        $this->factory = new Factory();
        $this->manager = new Manager($this->factory, $this->app);
    }

    /**
     * @test
     */
    public function it_can_create_a_manager_instance()
    {
        $this->assertInstanceOf(Manager::class, $this->manager);
    }

    /**
     * @test
     */
    public function it_can_register_a_new_theme()
    {
        $theme = $this->factory->fromPath(__DIR__ . '/stubs/parent/theme.json');

        $this->manager->register($theme, false);

        $this->assertTrue($this->manager->has('parent'));
    }

    /**
     * @test
     */
    public function it_will_not_overwrite_a_theme_with_the_same_name()
    {
        $theme1 = new Theme('test', 'path/1/', 'Test 1');
        $theme2 = new Theme('test', 'path/2/', 'Test 2');

        $this->manager->register($theme1, false);
        $this->manager->register($theme2, false);

        $this->assertSame($theme1->path(), $this->manager->get('test')->path());
    }

    /**
     * @test
     */
    public function it_can_register_a_new_theme_and_activate_it()
    {
        $theme = $this->factory->fromPath(__DIR__ . '/stubs/parent/theme.json');

        $this->manager->register($theme, true);

        $this->assertSame($theme, $this->manager->current());
    }

    /**
     * @test
     */
    public function it_can_register_a_new_theme_from_a_path()
    {
        $this->manager->registerPath(__DIR__ . '/stubs/parent/theme.json');

        $this->assertTrue($this->manager->has('parent'));
    }

    /**
     * @test
     */
    public function it_can_register_a_new_theme_from_a_path_and_activate_it()
    {
        $path = __DIR__ . '/stubs/parent/theme.json';

        $theme = $this->factory->fromPath(realpath($path));

        $this->manager->registerPath($path, true);

        $this->assertEquals($theme, $this->manager->current());
    }

    /**
     * @test
     */
    public function it_can_register_multiple_paths()
    {
        $this->manager->registerPaths([
            __DIR__ . '/stubs/parent/theme.json',
            __DIR__ . '/stubs/child/theme.json',
        ]);

        $this->assertCount(2, $this->manager->themes());
        $this->assertTrue($this->manager->has('parent'));
        $this->assertTrue($this->manager->has('child'));
    }

    /**
     * @test
     */
    public function it_can_filter_themes_by_name()
    {
        $this->manager->registerPaths([
            __DIR__ . '/stubs/parent/theme.json',
            __DIR__ . '/stubs/child/theme.json',
        ]);

        $theme = Arr::first($this->manager->filter('parent'));

        $this->assertSame('parent', $theme->name());
    }

    /**
     * @test
     */
    public function it_throws_the_correct_exception_when_it_cannot_find_a_theme_by_name()
    {
        $this->expectException(ThemeNotFoundException::class);

        $this->manager->get('foo');
    }

    /**
     * @test
     */
    public function it_can_activate_a_theme_using_only_a_name_if_that_theme_is_already_registered()
    {
        $this->manager->registerPath(__DIR__ . '/stubs/parent/theme.json');

        $this->manager->activate('parent');

        $this->assertSame('parent', $this->manager->current()->name());
    }

    /**
     * @test
     */
    public function it_can_register_the_views_with_the_view_finder()
    {
        $path = __DIR__ . '/stubs/parent';
        $this->manager->registerPath($path . '/theme.json');

        $this->manager->activate('parent');

        $this->assertContains(
            realpath($path) . '/views',
            $this->app['view']->getFinder()->getPaths()
        );
    }

    /**
     * @test
     */
    public function it_can_register_the_views_of_a_parent_theme_with_the_view_finder()
    {
        $pathChild = __DIR__ . '/stubs/child';
        $pathParent = __DIR__ . '/stubs/parent';

        $this->manager->registerPath($pathParent . '/theme.json');
        $this->manager->registerPath($pathChild . '/theme.json');

        $this->manager->activate('child');

        $this->assertContains(
            realpath($pathChild) . '/views',
            $this->app['view']->getFinder()->getPaths()
        );

        $this->assertContains(
            realpath($pathParent) . '/views',
            $this->app['view']->getFinder()->getPaths()
        );
    }

    /**
     * @test
     */
    public function it_can_discover_themes_in_a_path()
    {
        $files = $this->manager->discover(__DIR__ . '/stubs');

        $this->assertContains(__DIR__ . '/stubs/parent', $files);
        $this->assertContains(__DIR__ . '/stubs/child', $files);
    }
}
