<?php

namespace Ignite\Theme;

use Ignite\Theme\Exception\ThemeNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Routing\RouteCollection;

class UrlGenerator extends \Illuminate\Routing\UrlGenerator
{
    /**
     * @var Manager
     */
    protected $manager;

    /**
     * UrlGenerator constructor.
     *
     * @param RouteCollection $routes
     * @param Request $request
     * @param Manager $manager
     */
    public function __construct(RouteCollection $routes, Request $request, Manager $manager)
    {
        parent::__construct($routes, $request);
        $this->manager = $manager;
    }

    /**
     * Generate a URL to an application asset.
     *
     * @param  string  $path
     * @param  bool|null  $secure
     * @return string
     */
    public function asset($path, $secure = null)
    {
        if ($this->isValidUrl($path)) {
            return $path;
        }

        if ($this->request->isSecure()) {
            $secure = true;
        }

        // Once we get the root URL, we will check to see if it contains an index.php
        // file in the paths. If it does, we will remove it since it is not needed
        // for asset paths, but only for routes to endpoints in the application.
        $root = $this->formatRoot($this->formatScheme($secure));

        return $this->removeIndex($root) . $this->getThemeFileOrFallback($path);
    }

    /**
     * The path to the theme file.
     *
     * @param string $path
     * @param Theme $theme
     * @return string
     */
    protected function themePath($path, Theme $theme)
    {
        return '/themes/' . $theme->assetPath() . '/' . ltrim($path, '/');
    }

    /**
     * Get the correct theme URL depending upon the hierarchy.
     *
     * @param  string $url
     * @return string
     */
    protected function getThemeFileOrFallback($url)
    {
        /** @var Theme $current */
        $current = $this->manager->current();
        $themeFile = $this->themePath($url, $current);

        if (file_exists(public_path($themeFile)) || ! $current->hasParent()) {
            return $themeFile;
        }

        try {
            $parent = $this->manager->get($current->parent());
            $parentFile = $this->themePath($url, $parent);
        } catch (ThemeNotFoundException $e) {
            return $themeFile;
        }

        if (file_exists(public_path($parentFile))) {
            return $parentFile;
        }

        return $themeFile;
    }
}