<?php

namespace Ignite\Theme;

use Ignite\Theme\Exception\JsonDecodeException;
use Ignite\Theme\Exception\JsonNotFoundException;
use Illuminate\Support\Str;

class Factory
{
    /**
     * Create a Theme instance from a path.
     *
     * @param  string $path
     * @return Theme
     * @throws JsonNotFoundException
     * @throws JsonDecodeException
     */
    public function fromPath($path)
    {
        $path = rtrim($path, '/');

        if (! Str::endsWith($path, '/theme.json')) {
            $path = $path . '/theme.json';
        }

        if (! file_exists($path)) {
            throw new JsonNotFoundException($path);
        }

        $json = file_get_contents($path);

        return $this->fromJson($json, $path);
    }

    /**
     * Create a Theme instance from a path.
     *
     * @param  string $json
     * @param  string|null $path
     * @return Theme
     * @throws JsonDecodeException
     */
    public function fromJson($json, $path = null)
    {
        $theme = json_decode($json);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new JsonDecodeException(json_last_error_msg());
        }

        if (! empty($path)) {
            $theme->path = str_replace('/theme.json', '', $path);
        }

        return $this->create($theme);
    }

    /**
     * Create a Theme instance from Cache.
     *
     * @param  \stdClass $theme
     * @return Theme
     */
    public function create(\stdClass $theme)
    {
        return new Theme(
            $theme->name,
            $theme->path,
            $theme->description ?? null,
            $theme->parent ?? null
        );
    }
}
