<?php

namespace Ignite\Theme;

use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Support\Str;

class Theme implements Arrayable
{
    /**
     * The name of the theme.
     * @var string
     */
    protected $name;

    /**
     * The path to the theme.
     * @var string
     */
    protected $path;

    /**
     * A short description of the theme.
     * @var string
     */
    protected $description;

    /**
     * The parent theme.
     * @var string|null
     */
    protected $parent;

    /**
     * Theme constructor.
     *
     * @param string $name
     * @param string $path
     * @param string $description
     * @param string|null $parent
     */
    public function __construct($name, $path, $description = null, $parent = null)
    {
        $this->name = $name;
        $this->path = $path;
        $this->description = $description;
        $this->parent = $parent;
    }

    /**
     * The theme name.
     *
     * @return string
     */
    public function name()
    {
        return $this->name;
    }

    /**
     * The path to the theme.
     *
     * @return string
     */
    public function path()
    {
        return $this->path;
    }

    /**
     * The asset path to the theme.
     *
     * @return string
     */
    public function assetPath()
    {
        return Str::slug($this->name);
    }

    /**
     * The parent theme.
     *
     * @return string
     */
    public function parent()
    {
        return $this->parent;
    }

    /**
     * The theme description.
     *
     * @return string
     */
    public function description()
    {
        return $this->description;
    }

    /**
     * Determines whether or not a theme has a parent theme.
     *
     * @return bool
     */
    public function hasParent()
    {
        return ! empty($this->parent);
    }

    /**
     * Determines whether or not a theme is a child theme.
     *
     * @return bool
     */
    public function isChild()
    {
        return $this->hasParent();
    }

    /**
     * Get the theme as an array.
     *
     * @return array
     */
    public function toArray()
    {
        return get_object_vars($this);
    }
}
