<?php

namespace Ignite\Theme;

use Ignite\Theme\Exception\JsonDecodeException;
use Ignite\Theme\Exception\JsonNotFoundException;
use Ignite\Theme\Exception\ThemeNotFoundException;
use Illuminate\Contracts\Container\Container;
use Illuminate\Support\Arr;
use Illuminate\View\ViewFinderInterface;
use Webmozart\Glob\Iterator\GlobIterator;

class Manager
{
    /**
     * The registered themes.
     *
     * @var array Theme
     */
    protected $themes = [];

    /**
     * The currently active theme.
     *
     * @var Theme
     */
    protected $activeTheme;

    /**
     * Theme factory creates Theme objects.
     *
     * @var Factory
     */
    protected $factory;

    /**
     * @var Container
     */
    protected $app;

    /**
     * @var ViewFinderInterface
     */
    protected $view;

    /**
     * Manager constructor.
     *
     * @param Factory $factory
     * @param Container $app
     */
    public function __construct(Factory $factory, Container $app)
    {
        $this->factory = $factory;
        $this->app = $app;
        $this->view = $this->app->make('view');
    }

    /**
     * Filter the registered themes by the given theme name.
     *
     * @param  string $name
     * @return array
     */
    public function filter($name)
    {
        return array_filter($this->themes, function (Theme $theme) use ($name) {
            return $theme->name() === $name;
        });
    }

    /**
     * Checks to see whether a theme has been registered by the given name.
     *
     * @param  string $name
     * @return bool
     */
    public function has($name)
    {
        return ! empty($this->filter($name));
    }

    /**
     * Retrieves a theme based on its name. If no theme is found it'll throw a ThemeNotFoundException.
     *
     * @param  string $name
     * @return Theme
     * @throws ThemeNotFoundException
     */
    public function get($name)
    {
        $matches = $this->filter($name);

        if (empty($matches)) {
            throw new ThemeNotFoundException($name);
        }

        return Arr::first($matches);
    }

    /**
     * Returns an array of themes that have been registered.
     *
     * @return array
     */
    public function themes()
    {
        return $this->themes;
    }

    /**
     * Register a new theme based on its path. Optionally, activate the theme once registered.
     *
     * @param  Theme $theme
     * @param  bool $activate
     * @throws ThemeNotFoundException
     */
    public function register(Theme $theme, $activate = false)
    {
        if (! $this->has($theme->name())) {
            $this->themes[] = $theme;
        }

        if ($activate) {
            $this->activate($theme);
        }
    }

    /**
     * Register a theme based on its path.
     *
     * @param  string $path
     * @param  boolean $activate
     * @throws JsonDecodeException
     * @throws JsonNotFoundException
     * @throws ThemeNotFoundException
     */
    public function registerPath($path, $activate = false)
    {
        $theme = $this->factory->fromPath($path);

        $this->register($theme, $activate);
    }

    /**
     * Register a number of themes based on the array of paths provided.
     *
     * @param  array $paths
     * @throws JsonDecodeException
     * @throws JsonNotFoundException
     * @throws ThemeNotFoundException
     */
    public function registerPaths(array $paths)
    {
        foreach ($paths as $path) {
            $this->registerPath($path);
        }
    }

    /**
     * Searches for theme.json files within the directory structure specified by $directory and
     * returns the theme locations found. This method means that themes do not need to be manually
     * registered, however - it is a costly operation, and should be cached once you've found the
     * themes.
     *
     * @param  string $directory
     * @return array Returns an array of theme directory locations
     */
    public function discover($directory)
    {
        $iterator = new GlobIterator($directory . '/*/theme.json');
        $files = [];
        foreach ($iterator as $path) {
            $files[] = str_replace('/theme.json', '', $path);
        }

        return $files;
    }

    /**
     * Activate a theme. Activation can be done by the theme's name, or via a Theme object.
     *
     * @param  string|Theme $theme
     * @throws ThemeNotFoundException
     */
    public function activate($theme)
    {
        if (! $theme instanceof Theme) {
            $theme = $this->get($theme);
        }

        $this->activeTheme = $theme;

        $this->activateFinderPaths($theme);
    }

    /**
     * Activates the view finder paths for a theme and its parents.
     *
     * @param  Theme $theme
     * @throws ThemeNotFoundException
     */
    protected function activateFinderPaths(Theme $theme)
    {
        if ($theme->isChild()) {
            $this->activateFinderPaths($this->get($theme->parent()));
        }

        $this->registerViews($theme);
    }

    /**
     * Register views for the theme.
     *
     * @param  Theme $theme
     * @return $this
     */
    protected function registerViews(Theme $theme)
    {
        // We use `getFinder()->prependLocation()` because `addLocation()` will append.
        $this->app['view']->getFinder()->prependLocation($theme->path() . '/views');

        return $this;
    }

    /**
     * Returns the currently active theme.
     *
     * @return Theme
     */
    public function current()
    {
        return $this->activeTheme;
    }
}
