<?php
namespace Ignite\Theme\Tests;

use Ignite\Theme\Factory;
use Ignite\Theme\Manager;
use Ignite\Theme\Theme;
use Ignite\Theme\UrlGenerator;
use org\bovigo\vfs\vfsStream;
use org\bovigo\vfs\vfsStreamDirectory;

class UrlGeneratorTest extends TestCase
{
    /** @var Theme */
    protected $theme;

    /** @var Manager */
    protected $manager;

    /**
     * Setup the test environment.
     *
     * @return void
     */
    public function setUp()
    {
        parent::setUp();

        $this->theme = new Theme('test', __DIR__, 'The test theme.');
        $this->manager = new Manager(new Factory(), $this->app);
    }

    /**
     * @test
     */
    public function it_can_generate_the_correct_url_for_an_asset()
    {
        $this->manager->activate($this->theme);

        /** @var UrlGenerator $urlGenerator */
        $urlGenerator = $this->app->make(UrlGenerator::class, [
            'manager' => $this->manager
        ]);

        $this->assertSame(
            'http://localhost/themes/test/js/scripts.js',
            $urlGenerator->asset('js/scripts.js')
        );
    }

    /**
     * @test
     */
    public function it_can_generate_the_correct_secure_url_for_an_asset()
    {
        $this->manager->activate($this->theme);

        /** @var UrlGenerator $urlGenerator */
        $urlGenerator = $this->app->make(UrlGenerator::class, [
            'manager' => $this->manager
        ]);

        $this->assertSame(
            'https://localhost/themes/test/css/styles.css',
            $urlGenerator->asset('css/styles.css', true)
        );
    }

    /**
     * @test
     */
    public function it_will_return_the_path_when_the_url_is_invalid()
    {
        $this->manager->activate($this->theme);

        /** @var UrlGenerator $urlGenerator */
        $urlGenerator = $this->app->make(UrlGenerator::class, [
            'manager' => $this->manager
        ]);

        $this->assertSame(
            'https://https://localhost/themes/test/css/styles.css',
            $urlGenerator->asset('https://https://localhost/themes/test/css/styles.css', true)
        );
    }

    /**
     * @test
     */
    public function it_will_fallback_to_the_parent_theme_when_the_child_file_is_not_found()
    {
        $manager = $this->setUpDirectoryStream($path = __DIR__ . '/stubs', [
            'themes' => [],
            'public' => [
                'themes' => [
                    'parent' => [
                        'stylesheets' => [
                            'style.css' => ''
                        ]
                    ]
                ]
            ]
        ]);

        $manager->activate('child');

        /** @var UrlGenerator $urlGenerator */
        $urlGenerator = $this->app->make(UrlGenerator::class, [
            'manager' => $manager
        ]);

        $this->assertSame(
            'https://localhost/themes/parent/stylesheets/style.css',
            $urlGenerator->asset('stylesheets/style.css', true)
        );
    }

    /**
     * Set up the virtual directory structure.
     *
     * @param  string $directory
     * @return Manager
     * @throws \Ignite\Theme\Exception\JsonDecodeException
     * @throws \Ignite\Theme\Exception\JsonNotFoundException
     * @throws \Ignite\Theme\Exception\ThemeNotFoundException
     */
    protected function setUpDirectoryStream($directory, $structure = [])
    {
        $structure = $structure ?? [
            'themes' => [],
            'public' => [
                'themes' => []
            ]
        ];
        $this->root = vfsStream::setup('root', null, $structure);

        vfsStream::copyFromFileSystem($directory, $this->root->getChild('themes'));
        $this->app->setBasePath($this->root->url());

        // Assuming the following directories exist
        $this->app['config']->set('theme.paths', [
            vfsStream::url('root/themes'),
        ]);

        $manager = new Manager($this->app->make(Factory::class), $this->app);
        $manager->registerPaths(
            $manager->discover(
                vfsStream::url('root/themes')
            )
        );

        $this->app->instance(Manager::class, $manager);

        return $manager;
    }
}
