<?php

namespace Ignite\Theme\Tests\Console;

use Ignite\Theme\Factory;
use Ignite\Theme\Manager;
use Ignite\Theme\Tests\TestCase;
use Illuminate\Support\Facades\Event;
use org\bovigo\vfs\vfsStream;
use org\bovigo\vfs\vfsStreamDirectory;

class PublishCommandTest extends TestCase
{
    /**
     * @var vfsStreamDirectory
     */
    protected $root;

    /**
     * @var array
     */
    protected $paths = [];

    /**
     * @var null
     */
    protected $activate = null;

    /**
     * @test
     */
    public function it_publishes_all_themes_to_the_public_directory()
    {
        $this->setUpDirectoryStream(__DIR__ . '/../stubs');

        $command = $this->artisan('ignite:theme:publish');

        Event::shouldReceive('fire')->once()->with('ignite.theme.publishing');
        Event::shouldReceive('fire')->once()->with('ignite.theme.published');

        $command->expectsOutput('✓ Themes published.');
        $command->run();

        $this->assertDirectoryExists(public_path('themes/parent'));
        $this->assertDirectoryExists(public_path('themes/child'));
    }

    /**
     * @test
     */
    public function it_publishes_a_single_theme_to_the_public_directory()
    {
        $this->setUpDirectoryStream(__DIR__ . '/../stubs');

        $command = $this->artisan('ignite:theme:publish', [
            'theme' => 'parent'
        ]);

        Event::shouldReceive('fire')->once()->with('ignite.theme.publishing');

        $command->expectsOutput('✓ parent theme has been published.');

        $command->run();
        $this->assertDirectoryExists(public_path('themes/parent'));
    }

    /**
     * @test
     */
    public function it_throws_an_exception_when_the_distribution_directory_doesnt_exist()
    {
        $this->setUpDirectoryStream($path = __DIR__ . '/../stubs-no-dist');
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage("Missing `dist` folder in theme `no-dist`: vfs://root/themes/no-dist/dist");

        $command = $this->artisan('ignite:theme:publish', ['theme' => 'no-dist']);
        $command->run();

        Event::shouldReceive('fire')->once()->with('ignite.theme.publishing');
    }

    /**
     * Set up the virtual directory structure.
     *
     * @param  string $directory
     * @return Manager
     * @throws \Ignite\Theme\Exception\JsonDecodeException
     * @throws \Ignite\Theme\Exception\JsonNotFoundException
     * @throws \Ignite\Theme\Exception\ThemeNotFoundException
     */
    protected function setUpDirectoryStream($directory)
    {
        $this->root = vfsStream::setup('root', null, [
            'themes' => [],
            'public' => [
                'themes' => []
            ]
        ]);

        vfsStream::copyFromFileSystem($directory, $this->root->getChild('themes'));
        $this->app->setBasePath($this->root->url());

        // Assuming the following directories exist
        $this->app['config']->set('theme.paths', [
            vfsStream::url('root/themes'),
        ]);

        $manager = new Manager($this->app->make(Factory::class), $this->app);
        $manager->registerPaths(
            $manager->discover(
                vfsStream::url('root/themes')
            )
        );

        $this->app->instance(Manager::class, $manager);

        return $manager;
    }
}
