<?php

namespace Ignite\Theme\Tests;

use Ignite\Theme\Asset;
use Ignite\Theme\Factory;
use Ignite\Theme\Manager;
use Ignite\Theme\Theme;
use Ignite\Theme\UrlGenerator;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Str;

class AssetTest extends TestCase
{
    /** @var Asset */
    protected $asset;

    public function setUp()
    {
        parent::setUp();

        $this->asset = new Asset($this->app->make(Manager::class), $this->app->make(UrlGenerator::class));
    }

    /**
     * @test
     */
    public function it_can_create_a_manager_instance()
    {
        $this->assertInstanceOf(Asset::class, $this->asset);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_the_stylesheets_folder()
    {
        $style = $this->asset->style();

        $this->assertEquals(asset('themes/parent/stylesheets'), $style);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_a_stylesheet_file()
    {
        $style = $this->asset->style('/style.css');

        $this->assertEquals(asset('themes/parent/stylesheets/style.css'), $style);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_the_javascripts_folder()
    {
        $script = $this->asset->script();

        $this->assertEquals(asset('themes/parent/javascripts'), $script);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_a_javascript_file()
    {
        $script = $this->asset->script('/script.css');

        $this->assertEquals(asset('themes/parent/javascripts/script.css'), $script);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_the_images_folder()
    {
        $image = $this->asset->image();

        $this->assertEquals(asset('themes/parent/images'), $image);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_a_image_file()
    {
        $image = $this->asset->image('/logo.png');

        $this->assertEquals(asset('themes/parent/images/logo.png'), $image);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_the_fonts_folder()
    {
        $font = $this->asset->font();

        $this->assertEquals(asset('themes/parent/fonts'), $font);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_a_font_file()
    {
        $font = $this->asset->font('fontawesome-webfont.ttf');

        $this->assertEquals(asset('themes/parent/fonts/fontawesome-webfont.ttf'), $font);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_the_svg_folder()
    {
        $font = $this->asset->svg();

        $this->assertEquals(asset('themes/parent/svg'), $font);
    }

    /**
     * @test
     */
    public function it_can_generate_a_url_to_a_svg_file()
    {
        $svg = $this->asset->svg('test.svg');

        $this->assertEquals(asset('themes/parent/svg/test.svg'), $svg);
    }

    /**
     * @test
     */
    public function it_can_inline_a_svg_file()
    {
        $path = public_path('themes/parent/svg');
        $svg = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path class="heroicon-ui" d="M11 18.59V3a1 1 0 0 1 2 0v15.59l5.3-5.3a1 1 0 0 1 1.4 1.42l-7 7a1 1 0 0 1-1.4 0l-7-7a1 1 0 0 1 1.4-1.42l5.3 5.3z"/></svg>';

        if (! is_dir($path)) {
            mkdir($path, 0777, true);
        }

        $fp = fopen("{$path}/icon.svg", 'w+');
        fwrite($fp, $svg);
        fclose($fp);

        /** @var HtmlString $svg */
        $svg = $this->asset->inlineSvg('icon.svg', ['width' => 16, 'height' => 16, 'fill' => 'white']);

        $this->assertInstanceOf(HtmlString::class, $svg);
        $this->assertTrue(Str::startsWith($svg->toHtml(), '<svg width="16" height="16" fill="white"'));

        @unlink("{$path}/icon.svg");
    }

    /**
     * @test
     */
    public function it_can_be_extended_using_macros()
    {
        Asset::macro('svg', function ($path = null) {
            return $this->directoryOrFile('svg', $path);
        });

        $svgs = $this->asset->svg();
        $svg = $this->asset->svg('icon.svg');

        $this->assertEquals(asset('themes/parent/svg'), $svgs);
        $this->assertEquals(asset('themes/parent/svg/icon.svg'), $svg);
    }
}
