<?php

namespace Ignite\Theme;

use Illuminate\Contracts\Container\Container;
use Illuminate\Cache\Repository;

class CachedManager extends Manager
{
    /**
     * The cache key is used to determine the theme cache.
     *
     * @var string
     */
    protected $cacheKey = 'ignite.theme';

    /**
     * The cache repository.
     *
     * @var Repository
     */
    protected $cache;

    /**
     * Flag to determine whether the cache was hydrated.
     *
     * @var bool
     */
    protected $hydrated = false;

    /**
     * CachedManager constructor.
     *
     * @param Factory $factory
     * @param Container $app
     * @param Repository $cache
     */
    public function __construct(Factory $factory, Container $app, Repository $cache)
    {
        parent::__construct($factory, $app);
        $this->cache = $cache;
    }

    /**
     * Return the key used for cache storage.
     *
     * @return string
     */
    public function cacheKey()
    {
        return $this->cacheKey;
    }

    /**
     * Caches the registered themes, overwrite any existing cache.
     *
     * @param  array $themes
     * @return self
     */
    public function cache(array $themes = [])
    {
        $this->cache->forever($this->cacheKey, json_encode(array_map(function ($theme) {
            return $theme->toArray();
        }, $themes)));

        return $this;
    }

    /**
     * Clear any cached themes.
     *
     * @return self
     */
    public function clear()
    {
        $this->cache->forget($this->cacheKey);

        return $this;
    }

    /**
     * Hydrate the Theme data from the cache.
     *
     * @return self
     */
    public function hydrate()
    {
        if (! $this->cache->has($this->cacheKey)) {
            $this->hydrated = false;
            return $this;
        }

        $this->themes = array_map(function ($theme) {
            return $this->factory->fromJson($theme);
        }, $this->cache->get($this->cacheKey));

        $this->hydrated = true;

        return $this;
    }

    /**
     * Determine whether the manager has been hydrated from the cache.
     *
     * @return bool
     */
    public function hydrated()
    {
        return $this->hydrated;
    }
}
