<?php

namespace Ignite\Theme\Console;

use Ignite\Theme\Exception\ThemeNotFoundException;
use Ignite\Theme\Theme;
use Illuminate\Console\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

class PublishCommand extends Command
{
    const DIR_PUBLISH = 'dist';

    /**
     * @var string
     */
    protected $name = 'ignite:theme:publish';

    /**
     * @var string
     */
    protected $description = 'Publish assets associated with Ignite themes.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $themes = $this->getThemeManager()->themes();
        $this->getEventDispatcher()->fire('ignite.theme.publishing');

        $this->checkThemes($themes);

        if ($requestedTheme = $this->argument('theme')) {
            try {
                $theme = $this->getThemeManager()->get($requestedTheme);
                $this->publish($theme);
            } catch (ThemeNotFoundException $e) {
                $this->error($e->getMessage());
                $this->info('Please make you have added the theme path to your config file.');
            }
            return;
        }

        foreach ($themes as $theme) {
            $this->publish($theme);
        }

        $this->getEventDispatcher()->fire('ignite.theme.published');

        $this->info('<fg=green>✓</> Themes published.');
    }

    /**
     * Checks that the registered themes can be published.
     *
     * @param array $themes
     */
    protected function checkThemes(array $themes)
    {
        /** @var \Ignite\Theme\Theme $theme */
        foreach ($themes as $theme) {
            $this->checkTheme($theme);
        }
    }

    /**
     * Check that the given theme can be published.
     *
     * @param Theme $theme
     */
    protected function checkTheme(Theme $theme)
    {
        if (! $this->getFilesystemManager()->exists($path = $this->getThemeDistributionPath($theme))) {
            throw new \InvalidArgumentException("Missing `dist` folder in theme `{$theme->name()}`: " . $path);
        }
    }

    /**
     * Publish a single theme's assets.
     *
     * @param Theme $theme
     */
    protected function publish(Theme $theme)
    {
        $source = $this->getThemeDistributionPath($theme);
        $destination = $this->option('path') . DIRECTORY_SEPARATOR . 'themes' . DIRECTORY_SEPARATOR . $theme->assetPath();

        $this->getFilesystemManager()->copyDirectory($source, $destination);

        $this->info("<fg=green>✓</> {$theme->name()} theme has been published.");
    }

    /**
     * The Theme manager class.
     *
     * @return \Ignite\Theme\Manager
     */
    protected function getThemeManager()
    {
        return $this->laravel['theme'];
    }

    /**
     * The path to the distribution files to be copied to public.
     *
     * @param Theme $theme
     * @return string
     */
    protected function getThemeDistributionPath(Theme $theme)
    {
        return $theme->path() . DIRECTORY_SEPARATOR . static::DIR_PUBLISH;
    }

    /**
     * The Filesystem manager class.
     *
     * @return \Illuminate\Filesystem\Filesystem
     */
    protected function getFilesystemManager()
    {
        return $this->laravel['files'];
    }

    /**
     * The Event dispatcher class.
     *
     * @return \Illuminate\Events\Dispatcher
     */
    protected function getEventDispatcher()
    {
        return $this->laravel['events'];
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    public function getArguments()
    {
        return [
            ['theme', InputArgument::OPTIONAL, 'Name of the theme to publish']
        ];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    public function getOptions()
    {
        return [
            ['path', null, InputOption::VALUE_OPTIONAL, "The path to the directory the theme should be published.", public_path()],
        ];
    }
}
