(function ($, window, undefined) {
    'use strict';

    // Create the defaults once
    var pluginName = 'conditionals',
        defaults = {
            /**
             * This callback type is called `onInit` and offers the plugin caller a chance to run custom code
             * on plugin initialization.
             *
             * @type Function
             * @callback onInit
             */
            onInit: function () {},

            /**
             * An array of Rule objects.
             *
             * @type Object[] e.g.
             * {
             *     name: 'field_name',
             *     triggers: ['field_name'],
             *     fields: [
             *         'field_name',
             *     ],
             *     conditions: [
             *         {
             *             checks: [
             *                 {
             *                     field: 'field_name',
             *                     op: '=',
             *                     value: 'Field Value'
             *                 }
             *             ],
             *             show: [
             *                 'field_name',
             *             ],
             *             required: [
             *                 'field_name',
             *             ],
             *             onHandle: function (condition) {}
             *         }
             *     ]
             * }
             */
            rules: [],

            /**
             * The label to show when we replace options and a selected item was not found.
             *
             * @type String
             */
            emptyLabel: '--- select ---',

            /**
             * Resolve a jQuery element node using a callback, by default we lookup an id.
             *
             * @param identifier
             * @returns {jQuery|HTMLElement}
             */
            elementResolver: function (identifier) {
                return $('#' + identifier);
            }
        };

    /**
     * The actual plugin constructor.
     *
     * @param {Node} element
     * @param {Object} options
     */
    function Plugin(element, options) {
        this.element = element;
        this.settings = $.extend({}, defaults, options);
        this._defaults = defaults;
        this._name = pluginName;
        this.init();
    }

    // Avoid Plugin.prototype conflicts
    $.extend(Plugin.prototype, {

        /**
         * Initialize the rules and trigger necessary changes on load.
         */
        init: function () {
            if (this.settings.hasOwnProperty('rules') && this.settings.rules.length) {
                for (var i = 0; i < this.settings.rules.length; i++) {
                    this.initRule(this.settings.rules[i]);
                    this.triggerRule(null, this.settings.rules[i]);
                }
            }

            this.settings.onInit.apply(this);
        },

        /**
         * Initialize a individual rule and register necessary onChange functionality.
         *
         * @param {Object} rule
         */
        initRule: function (rule) {
            var conditionals = this;
            var onChange = function (e) {
                conditionals.triggerRule(e, rule);
            };

            if (rule.hasOwnProperty('triggers') && rule.triggers.length) {
                for (var i = 0; i < rule.triggers.length; i++) {
                    var node = this.elementResolver(rule.triggers[i]);
                    if (node) {
                        node.on('change', onChange);
                    }
                }
            }
        },

        /**
         * Trigger an individual rule.
         *
         * @param {Object} event
         * @param {Object} rule
         */
        triggerRule: function (event, rule) {
            var shownFields = [];

            if (rule.hasOwnProperty('conditions') && rule.conditions.length) {
                for (var i = 0; i < rule.conditions.length; i++) {
                    var fields = this.handleCondition(rule.conditions[i]);
                    shownFields = shownFields.concat(fields);
                }
            }

            if (rule.hasOwnProperty('fields') && rule.fields.length) {
                // hide/empty all fields not shown
                for (var j = 0; j < rule.fields.length; j++) {
                    if (shownFields.indexOf(rule.fields[j]) === -1) {
                        this.hideField(rule.fields[j], true);
                    }
                }
            }
        },

        /**
         * Handle an individual condition.
         *
         * @param {Object} condition
         * @return Array
         */
        handleCondition: function (condition) {
            var isMet = true;

            if (condition.hasOwnProperty('checks') && condition.checks.length) {
                for (var i = 0; i < condition.checks.length; i++) {
                    isMet = this.handleCheck(condition.checks[i]) && isMet;
                }
            }

            if (! isMet) {
                return [];
            }

            // show fields
            if (condition.hasOwnProperty('show') && condition.show.length) {
                for (var j = 0; j < condition.show.length; j++) {
                    this.showField(condition.show[j]);
                }
            }

            // required fields
            if (condition.hasOwnProperty('required') && condition.required.length) {
                for (var k = 0; k < condition.required.length; k++) {
                    this.requireField(condition.required[k]);
                }
            }

            // custom handle
            if (condition.hasOwnProperty('onHandle') && typeof condition.onHandle === 'function') {
                condition.onHandle.apply(this, [condition]);
            }

            return condition.show || false;
        },

        /**
         * Handle an individual check.
         *
         * @param {Object} check
         * @return Boolean
         */
        handleCheck: function (check) {
            var node = this.elementResolver(check.field);
            var value = null;

            if (! node) {
                return false;
            }

            // In display only mode, you need to find the input inside the given div node
            if (node.hasClass('displayOnly') || node.prop('readonly')) {
                value = $('input', node).val();
            } else {
                value = node.val();
            }

            // TODO: Support more operation types
            if ($.inArray(check.op, ['=', 'in', 'notIn']) === -1) {
                throw 'Unrecognized op: ' + check.op;
            }

            if (typeof check.value === 'undefined') {
                throw 'Condition based on value is the only option currently implemented.';
            }

            if ('in' === check.op) {
                return $.inArray(value, check.value) !== -1;
            }

            if ('notIn' === check.op) {
                return $.inArray(value, check.value) === -1;
            }

            return (check.value == value);
        },

        /**
         * Show an individual field.
         *
         * @param {String} name
         */
        showField: function (name) {
            var n = this.elementResolver(name);
            if (n) {
                n.show();
                // get 2 divs up and show row
                // todo: make this a little more open to fields with a different structure,
                //  e.g. find the parent with a child element of a given class name: n.parent('.form-group')
                n.closest('.form-group').show();
            }
        },

        /**
         * Set an individual field as required.
         *
         * @param {String} name
         */
        requireField: function (name) {
            var n = this.elementResolver(name);
            if (n) {
                n.prop('required', true);
            }
        },

        /**
         * Set an individual field as required.
         *
         * @param {String} name
         * @param {Boolean=} isEmpty
         */
        hideField: function (name, isEmpty) {
            var n = this.elementResolver(name);
            if (n) {
                n.hide();

                // in display only mode, you need to find the input inside the given div node
                if (n.hasClass('displayOnly')) {
                    // get 1 divs up and hide row
                    n.closest('.form-group').hide();
                } else {
                    n.prop('required', false);
                    // get 2 divs up and hide row
                    n.closest('.form-group').hide();
                    if (isEmpty) {
                        n.val('');
                    }
                }
            }
        },

        /**
         * Toggle the element type from text input to select dropdown.
         *
         * @param {jQuery|HTMLElement|Node} element
         * @param {Array} options
         * @param {String} value
         */
        toggleInputSelectOptions: function (element, options, value) {
            if (element.prop('tagName') === 'INPUT') {
                this.convertInputToSelect(element, options, value || element.val() || element.attr('data-initial'));
            } else {
                this.replaceSelect2Options(element, options, value || element.val() || element.attr('data-initial'));
            }
        },

        /**
         * Toggle the element type from select dropdown to text input.
         *
         * @param {jQuery|HTMLElement|Node} element
         * @param {String} value
         */
        toggleSelectOptionsInput: function (element, value) {
            if (element.prop('tagName') === 'SELECT') {
                this.convertSelectToInput(element, value || element.val() || element.attr('data-initial'));
            }
        },

        /**
         * Replace the options within a select2 component.
         *
         * @param {jQuery|HTMLElement|Node} node
         * @param {Array} options
         * @param {String} value
         */
        replaceSelect2Options: function (node, options, value) {
            // if is displayOnly with input instead of select2
            // then no need to re-do the options,
            // (which would convert it into select2 from displayOnly)
            if (node.hasClass('displayOnly') || node.prop('readonly')) {
                return;
            }

            var initialValue = node.attr('data-initial');
            var oldValue = node.val();

            node.select2().empty();
            node.val(null).trigger('change');

            this.initializeSelect2(
                node,
                options,
                (typeof value !== 'undefined' ? value : '') || oldValue || initialValue
            );
        },

        /**
         * Resolve the element given the identifier. By default we append the identifier to '#' for an ID.
         *
         * @param {String} identifier
         */
        elementResolver: function (identifier) {
            return this.settings.elementResolver.call(this, [identifier]);
        },

        /**
         * Convert a select form field into a text input field.
         *
         * @param {jQuery|HTMLElement} node
         * @param {String} value
         * @returns {jQuery|HTMLElement}
         */
        convertSelectToInput: function (node, value) {
            if (node.prop('tagName') === 'SELECT') {
                var $input = $('<input />', {
                    'type': 'text',
                    'id': node.attr('id') || node.attr('name'),
                    'class': 'form-control',
                    'name': node.attr('name'),
                    'data-initial': node.attr('data-initial'),
                    'value': value
                });

                if (node.prop('required')) {
                    $input.prop('required', true);
                }

                node.parent().append($input);

                if (node.hasClass('select2-hidden-accessible')) {
                    node.select2('destroy');
                }

                node.remove();
            }

            return node;
        },

        /**
         * Convert a input text field into a select form field.
         *
         * @param {jQuery|HTMLElement} node
         * @param {Array} options
         * @returns {*}
         */
        convertInputToSelect: function (node, options) {
            if (node.prop('tagName') === 'INPUT' && node.attr('type').toUpperCase() === 'TEXT') {
                var $select = $('<select />', {
                    'id': node.attr('id') || node.attr('name'),
                    'class': 'form-control',
                    'name': node.attr('name'),
                    'data-initial': node.attr('data-initial'),
                    'value': node.val()
                });

                node.parent().append($select);

                $select = this.initializeSelect2($select, options, node.val() || node.attr('data-initial'));

                node.remove();
            }

            return node;
        },

        initializeSelect2: function (node, options, value) {
            var data = []; // Temporary storage array for option objects
            var found = false; // set previous value again if available

            data.push({id: '', text: this.settings.emptyLabel});

            for (var i in options) {
                if (options.hasOwnProperty(i)) {
                    data.push({id: i, text: options[i] });
                    console.log('key', i);
                    console.log('val', value);
                    if (i == value) {
                        found = true;
                    }
                }
            }

            console.log(found);

            // Re-create the select2 component
            node.select2({
                data: data,         // Add options
                theme: 'bootstrap'  // Add bootstrap theme
            });

            if (found) {
                node.val(value).trigger('change');
            }

            return node;
        }
    });

    /**
     * A really lightweight plugin wrapper around the constructor,
     * preventing against multiple instantiations.
     *
     * @param {Object} options
     */
    $.fn[pluginName] = function (options) {
        return this.each(function () {
            if (!$.data(this, 'plugin_' + pluginName)) {
                $.data(this, 'plugin_' + pluginName, new Plugin(this, options));
            }
        });
    };

})(jQuery, window);
