<?php

namespace Ignite\Vendor\Tango;

use GuzzleHttp\Client as Guzzle;
use Ignite\Vendor\Tango\Request\OrderRequest;
use Ignite\Vendor\Tango\Request\Request;
use Ignite\Vendor\Tango\Response\ErrorResponse;
use Ignite\Vendor\Tango\Response\OrderResponse;
use Psr\Http\Message\ResponseInterface;

class Client
{
    /**
     * @var string
     */
    protected $platformName;

    /**
     * @var string
     */
    protected $apiKey;

    /**
     * @var Client
     */
    protected $client;

    /**
     * The production endpoint.
     *
     * @var string
     */
    protected $liveEndpoint = 'https://api.tangocard.com';

    /**
     * The testing endpoint.
     *
     * @var string
     */
    protected $testEndpoint = 'https://integration-api.tangocard.com';

    /**
     * Flag to indicate if we are in test mode.
     *
     * @var bool
     */
    protected $testMode = false;

    /**
     * TangoApi constructor.
     *
     * @param string      $username
     * @param string      $apiKey
     * @param Guzzle|null $client
     * @param bool        $testMode
     */
    public function __construct(string $username, string $apiKey, Guzzle $client = null, $testMode = false)
    {
        $this->platformName = $username;
        $this->apiKey = $apiKey;
        $this->testMode = (bool) $testMode;
        $this->client = is_null($client) ? $this->getDefaultClient() : $client;
    }

    /**
     * Submit an order.
     *
     * @param OrderRequest $request
     *
     * @return OrderResponse|ErrorResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function postOrder(OrderRequest $request)
    {
        $method = '/raas/v2/orders';
        $response = $this->buildPostRequest($method, $request);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return OrderResponse::fromHttpResponse($response);
    }

    /**
     * Build a POST request.
     *
     * @param string  $method
     * @param Request $request
     *
     * @return ResponseInterface
     * @throws GuzzleException
     */
    protected function buildPostRequest($method, Request $request)
    {
        return $this->client->request('post', $method, [
            'json' => $request->toArray(),
            'auth' => [$this->platformName, $this->apiKey],
            'headers' => $this->getDefaultHeaders(),
        ]);
    }

    /**
     * Build a GET request.
     *
     * @param string $method
     * @param array  $query
     *
     * @return ResponseInterface
     * @throws GuzzleException
     */
    protected function buildGetRequest($method, $query, $requestId)
    {
        return $this->client->request('GET', $method, [
            'query' => $query,
            'auth' => [$this->platformName, $this->apiKey],
            'headers' => $this->getDefaultHeaders(),
        ]);
    }

    /**
     * The default headers and possibly merge in additional headers.
     *
     * @param array $additional
     * @return array
     */
    protected function getDefaultHeaders($additional = [])
    {
        $defaults = [
            // 'Authorization' => 'Bearer ' . base64_encode($this->platformName . ':' . $this->apiKey),
        ];
        return array_merge($defaults, $additional);
    }

    /**
     * The default configured HTTP client.
     *
     * @param array $options
     *
     * @return Guzzle
     */
    protected function getDefaultClient($options = [])
    {
        return new Guzzle(array_merge([
            'base_uri' => $this->testMode === false ? $this->liveEndpoint : $this->testEndpoint,
            'debug' => $this->testMode === false ? false : true,
            'verify' => false,
            'exceptions' => false,
            // 'curl' => [
            //     CURLOPT_SSL_VERIFYPEER => true,
            //     CURLOPT_SSLVERSION => CURL_SSLVERSION_TLSv1_2,
            // ]
        ], $options));
    }

    /**
     * Determine if the response status code is an error code.
     *
     * @param ResponseInterface $response
     *
     * @return bool
     */
    protected function isErrorCode(ResponseInterface $response)
    {
        return in_array($response->getStatusCode(), [400, 404, 500, 502, 503]);
    }
}
