<?php

namespace Ignite\StateMachine;

use Ignite\StateMachine\Contracts\FactoryInterface;
use Ignite\StateMachine\Contracts\StatefulInterface;
use Illuminate\Contracts\Events\Dispatcher;

class Factory implements FactoryInterface
{
    /**
     * @var array
     */
    private $states;

    /**
     * @var array
     */
    private $transitions;

    /**
     * @var Dispatcher
     */
    private $dispatcher;

    /**
     * Create a configured StateMachine instance.
     *
     * @param array $states
     * @param array $transitions
     * @param Dispatcher $dispatcher
     *
     * @return void
     */
    public function __construct(array $states, array $transitions, Dispatcher $dispatcher)
    {
        $this->states = $states;
        $this->transitions = $transitions;
        $this->dispatcher = $dispatcher;
    }

    /**
     * Create the state machine instance.
     *
     * @param StatefulInterface $stateful
     *
     * @return StateMachine
     */
    public function create(StatefulInterface $stateful)
    {
        $transitions = $this->getTransitionInstances();
        $states = $this->getStateInstances($transitions);

        return new StateMachine($stateful, $states, $transitions, $this->dispatcher);
    }

    /**
     * Create the state instances and their associated transitions.
     *
     * @param array $transitions
     *
     * @return array
     */
    protected function getStateInstances($transitions)
    {
        $states = [];

        foreach ($this->states as $key => $state) {
            $name = isset($state['name']) ? $state['name'] : $key;
            $states[$name] = new State($name, $state['type']);
        }

        foreach ($transitions as $transition) {
            foreach ($transition->initialStates() as $state) {
                $states[$state]->addTransition($transition);
            }
        }

        return $states;
    }

    /**
     * Create the transition instances.
     *
     * @return array
     */
    protected function getTransitionInstances()
    {
        $transitions = [];

        foreach ($this->transitions as $name => $transition) {
            $transitionClass = isset($transition['class']) ? $transition['class'] : Transition::class;
            $transitions[$name] = new $transitionClass(
                $name,
                $transition['from'],
                $transition['to'],
                isset($transition['guard']) ? $transition['guard'] : null
            );
        }

        return $transitions;
    }
}
