<?php

namespace Ignite\StateMachine\Contracts;

use Ignite\StateMachine\Exceptions\StateNotFoundException;
use Ignite\StateMachine\Exceptions\TransitionNotAllowedException;
use Ignite\StateMachine\Exceptions\TransitionNotFoundException;

interface StateMachineInterface
{
    /**
     * The stateful instance.
     *
     * @return StatefulInterface
     */
    public function getStatefulInstance();

    /**
     * The current state.
     *
     * @return StateInterface
     */
    public function state();

    /**
     * Find a transition by name.
     *
     * @param string $name
     *
     * @return TransitionInterface
     * @throws TransitionNotFoundException
     */
    public function getTransition($name);

    /**
     * The registered transitions.
     *
     * @return array
     */
    public function getTransitions();

    /**
     * Find a state by name.
     *
     * @param string $name
     *
     * @return StateInterface
     * @throws StateNotFoundException
     */
    public function getState($name);

    /**
     * The registered states.
     *
     * @return array
     */
    public function getStates();

    /**
     * The initial state for the state machine.
     *
     * @return StateInterface|null
     */
    public function getInitialState();

    /**
     * Determine whether the current state can be 'transitioned' to another state using the given transition.
     *
     * @param string $name
     * @param array $context
     *
     * @return bool
     */
    public function can($name, $context = []);

    /**
     * Apply the given transition.
     *
     * @param string $name
     * @param array $context
     *
     * @return mixed
     * @throws TransitionNotFoundException
     * @throws TransitionNotAllowedException
     * @throws StateNotFoundException
     */
    public function apply($name, $context = []);

    /**
     * Dispatch a transition event with the context of the state machine
     *
     * @param string|object $name
     * @param TransitionInterface $transition
     * @param array $context
     * @param StatefulInterface|null $stateful
     *
     * @return array|null
     */
    public function dispatchTransitionEvent(
        $name,
        TransitionInterface $transition,
        array $context = [],
        StatefulInterface $stateful = null
    );
}
