<?php

namespace Ignite\StateMachine;

use Ignite\StateMachine\Contracts\StateMachineInterface;
use Ignite\StateMachine\Exceptions\TransitionNotFoundException;
use Illuminate\Support\Str;

trait FiniteStateMachineTrait
{
    /**
     * The singleton property to hold the state machine instance.
     *
     * @var StateMachineInterface
     */
    protected $stateMachine;

    /**
     * The flag to determine whether we should always load the state machine when hydrating and saving.
     *
     * @var bool
     */
    protected $withStateMachine = true;

    /**
     * Use the model with the state machine.
     *
     * @param bool $flag
     *
     * @return $this
     */
    public function useWithStateMachine($flag = true)
    {
        $this->withStateMachine = $flag;

        return $this;
    }

    /**
     * Use the model without the state machine.
     *
     * @param bool $flag
     *
     * @return $this
     */
    public function useWithoutStateMachine($flag = true)
    {
        $this->withStateMachine = $flag;

        return $this;
    }

    /**
     * Boot the trait on the eloquent model.
     *
     * @return void
     */
    public static function bootFiniteStateMachineTrait()
    {
        static::retrieved(function ($model) {
            if ($model->withStateMachine) {
                $model->stateMachine();
            }
        });

        static::saving(function ($model) {
            // We use `isDirty()` instead of `wasChanged()` since we are
            // checking before the database command execution is run.
            if ($model->withStateMachine && $model->isDirty($model->getStateMachineAttribute())) {
                $model->stateMachine()->apply($model->getTransitionName());
            }
        });
    }

    /**
     * The default state machine config should return an array with the following structure:
     *
     * [
     *     'states' => [
     *         'pending' => [
     *             'type' => StateInterface::INITIAL,
     *         ],
     *     ],
     *     'transitions' => [
     *         'activate' => [
     *             'from' => ['pending'],
     *             'to' => 'active',
     *             'guard' => function ($sm) { return true; }
     *         ],
     *     ]
     * ]
     *
     * @return array
     */
    abstract public function getStateMachineConfig();

    /**
     * The attribute we should monitor for state changes. E.g. 'status'.
     *
     * @return string
     */
    abstract public function getStateMachineAttribute();

    /**
     * The state machine instance for the current eloquent model.
     *
     * @return StateMachineInterface
     */
    public function stateMachine()
    {
        if (! $this->stateMachine) {
            $this->stateMachine = app(Factory::class, $this->getStateMachineConfig())->create($this);
        }

        return $this->stateMachine;
    }

    /**
     * Set the state value.
     *
     * @param string $state
     */
    public function setState(string $state)
    {
        $this->setAttribute($this->getStateMachineAttribute(), $state);
    }

    /**
     * Get the state value.
     *
     * @return string
     */
    public function getState()
    {
        return $this->getAttribute($this->getStateMachineAttribute());
    }

    /**
     * Determine the transition name from the given states change.
     *
     * @return mixed
     * @throws Exceptions\TransitionNotFoundException
     */
    public function getTransitionName()
    {
        $attribute = $this->getStateMachineAttribute();
        $transitions = $this->getTransitionMap();
        $name = $this->formatTransitionName(
            $this->getOriginal($attribute) ?? $this->stateMachine()->getInitialState()->name(),
            $this->getAttribute($attribute)
        );


        if (! isset($transitions[$name])) {
            throw new TransitionNotFoundException(
                $name,
                $this->stateMachine()->getTransitions(),
                $this
            );
        }

        return $transitions[$name];
    }

    /**
     * The mapping of '{state-from}-{state-to}' => 'transition'.
     *
     * @return array
     */
    protected function getTransitionMap()
    {
        $transitions = [];

        /** @var Transition $transition */
        foreach ($this->stateMachine->getTransitions() as $transition) {
            foreach ($transition->initialStates() as $state) {
                $transitions[$this->formatTransitionName($state, $transition->state())] = $transition->name();
            }
        }

        return $transitions;
    }

    /**
     * The formatted transition name. E.g. 'pending-active'.
     *
     * @param string $old
     * @param string $new
     *
     * @return string
     */
    protected function formatTransitionName($old, $new)
    {
        return Str::slug($old) . '-' . Str::slug($new);
    }
}
