<?php

require "vendor/autoload.php";

use Ignite\StateMachine\Contracts\StateMachineInterface;
use Ignite\StateMachine\Contracts\StatefulInterface;
use Ignite\StateMachine\Contracts\StateInterface;
use Ignite\StateMachine\Exceptions\TransitionNotFoundException;
use Ignite\StateMachine\State;
use Ignite\StateMachine\StateMachine;
use Ignite\StateMachine\Transition;
use Illuminate\Container\Container;
use Illuminate\Events\Dispatcher;

trait StatefulTrait
{
    /**
     * @var string
     */
    protected $state = 'submitted';

    /**
     * @inheritDoc
     */
    public function setState(string $state)
    {
        $this->state = $state;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getState()
    {
        return $this->state;
    }
}

interface ClaimLogInterface
{
    public function error($message, array $context = []);
}

class Claim implements StatefulInterface, ClaimLogInterface
{
    use StatefulTrait;

    /**
     * @var string
     */
    protected $country = 'CA';

    /**
     * @var array
     */
    protected $log = [];

    /**
     * The 2-letter country code.
     *
     * @return string
     */
    public function getCountry(): string
    {
        return $this->country;
    }

    public function error($message, array $context = [])
    {
        $this->log['error'] = compact('message', $context);
    }
}

class NoCanadiansGuard
{
    public function __invoke($machine)
    {
        $stateful = $machine->getStatefulInstance();
        $country = $stateful->getCountry();

        $isCanadian = ($country === 'CA');

        if ($isCanadian && $stateful instanceof ClaimLogInterface) {
            $stateful->error('Claim is associated with Canada. #nocanadians #blamecanada', compact('stateful'));
        }

        return ! $isCanadian;
    }
}

class CustomTransition extends Transition
{
    /**
     * @inheritDoc
     */
    public function process(StateMachineInterface $stateMachine)
    {
        return $stateMachine->getStatefulInstance();
    }
}

$stateful = new Claim();

$config = [
    'states' => [
        'submitted' => [
            'type' => StateInterface::INITIAL,
        ],
        'declined' => [
            'type' => StateInterface::FINAL,
        ],
        'approved' => [
            'type' => StateInterface::NORMAL,
        ],
        'cancelled' => [
            'type' => StateInterface::FINAL,
        ],
        'issued' => [
            'type' => StateInterface::FINAL,
        ]
    ],
    'transitions' => [
        'approve' => [
            'from' => ['pending', 'submitted'],
            'to' => 'approved',
            'guard' => new NoCanadiansGuard(),
            'class' => CustomTransition::class,
        ],
        'decline' => [
            'from' => ['submitted'],
            'to' => 'declined',
        ],
        'cancel' => [
            'from' => ['approved'],
            'to' => 'cancelled',
        ],
        'issue' => [
            'from' => ['approved'],
            'to' => 'issued',
            'guard' => new NoCanadiansGuard(),
        ]
    ],
];

$states = [];
foreach ($config['states'] as $name => $state) {
    $states[$name] = new State($name, $state['type'], []);
}

$transitions = [];
foreach ($config['transitions'] as $name => $transition) {
    //$transition['guard']
    $className = $transition['class'] ?? Transition::class;
    $transitions[$name] = new $className($name, $transition['from'], $transition['to'], $transition['guard'] ?? null);
}

//var_dump($states);
//var_dump($transitions);

foreach ($transitions as $transition) {
    foreach ($transition->initialStates() as $state) {
        $states[$state]->addTransition($transition);
    }
}

$dispatcher = new Dispatcher(new Container());

$stateMachine = new StateMachine($stateful, $states, $transitions, $dispatcher);

//die(var_dump($stateMachine->getStates()));
//die(var_dump($stateMachine->getTransitions()));

//die(var_dump($stateMachine->can('issue')));
var_dump($stateMachine->apply('approve'));
//var_dump($stateMachine->getStatefulInstance()->getState());
//var_dump($stateMachine->state()->name());

//try {
//    var_dump($stateMachine->apply('issue'));
//    var_dump($stateMachine->getStatefulInstance()->getState());
//    var_dump($stateMachine->state()->name());
//} catch (TransitionNotFoundException $e) {
//
//}
