<?php

namespace Ignite\StateMachine;

use Ignite\StateMachine\Contracts\StateInterface;
use Ignite\StateMachine\Contracts\TransitionInterface;

class State implements StateInterface
{
    /**
     * @var string
     */
    protected $name;

    /**
     * @var string
     */
    protected $type;

    /**
     * @var array
     */
    protected $transitions;

    /**
     * State constructor.
     *
     * @param string $name
     * @param string $type
     * @param array $transitions
     * @param callable|null $guard
     */
    public function __construct(string $name, string $type, array $transitions = [])
    {
        $this->name = $name;
        $this->type = $type;
        $this->setTransitions($transitions);
    }

    /**
     * Is the status in its initial state.
     *
     * @return bool
     */
    public function isInitial()
    {
        return $this->type === StateInterface::INITIAL;
    }

    /**
     * Is the status in its normal state.
     *
     * @return bool
     */
    public function isNormal()
    {
        return $this->type === StateInterface::NORMAL;
    }

    /**
     * Is the status in its final state.
     *
     * @return bool
     */
    public function isFinal()
    {
        return $this->type === StateInterface::FINAL;
    }

    /**
     * The name of the transition.
     *
     * @return string
     */
    public function name()
    {
        return $this->name;
    }

    /**
     * The state type representation.
     *
     * @return string
     */
    public function type()
    {
        return $this->type;
    }

    /**
     * The previous transitions during the current request.
     *
     * @return array
     */
    public function transitions()
    {
        return $this->transitions;
    }

    /**
     * Add a transition.
     *
     * @param TransitionInterface $transition
     *
     * @return $this
     */
    public function addTransition(TransitionInterface $transition)
    {
        $this->transitions[$transition->name()] = $transition;

        return $this;
    }

    /**
     * Set the array of transitions.
     *
     * @param array $transitions
     *
     * @return $this
     */
    protected function setTransitions(array $transitions)
    {
        foreach ($transitions as $transition) {
            $this->transitions[$transition->name()] = $transition;
        }

        return $this;
    }
}
