<?php

namespace Ignite\StateMachine\Exceptions;

use Exception;
use Ignite\StateMachine\Contracts\StatefulInterface;
use Ignite\StateMachine\Contracts\StateInterface;
use Ignite\StateMachine\Contracts\TransitionInterface;
use Throwable;

class TransitionNotAllowedException extends Exception
{
    /**
     * @var TransitionInterface
     */
    protected $transition;

    /**
     * @var StateInterface
     */
    protected $state;

    /**
     * @var StatefulInterface
     */
    protected $stateful;

    /**
     * TransitionNotAllowedException constructor.
     *
     * @param TransitionInterface $transition
     * @param StateInterface $state
     * @param StatefulInterface $stateful
     * @param int $code
     * @param Throwable|null $previous
     */
    public function __construct(
        TransitionInterface $transition,
        StateInterface $state,
        StatefulInterface $stateful,
        $code = 0,
        Throwable $previous = null
    ) {
        $this->transition = $transition;
        $this->stateful = $stateful;
        $this->state = $state;

        parent::__construct($this->formatMessage(), $code, $previous);
    }

    /**
     * The formatted exception message.
     *
     * @return string
     */
    protected function formatMessage()
    {
        return sprintf(
            'Cannot transition %s from `%s` to `%s`',
            get_class($this->stateful),
            $this->state->name(),
            $this->transition->state()
        );
    }

    /**
     * Get the transition.
     *
     * @return TransitionInterface
     */
    public function getTransition()
    {
        return $this->transition;
    }

    /**
     * Get the state.
     *
     * @return StateInterface
     */
    public function getState()
    {
        return $this->state;
    }

    /**
     * Get the stateful object instance.
     *
     * @return StatefulInterface
     */
    public function getStatefulInstance()
    {
        return $this->stateful;
    }
}
