<?php

namespace Ignite\StateMachine\Tests;

use Ignite\StateMachine\Contracts\StatefulInterface;
use Ignite\StateMachine\Contracts\StateInterface;
use Ignite\StateMachine\Contracts\TransitionInterface;
use Ignite\StateMachine\Exceptions\StateNotFoundException;
use Ignite\StateMachine\Exceptions\TransitionNotFoundException;
use Ignite\StateMachine\State;
use Ignite\StateMachine\StateMachine;
use Ignite\StateMachine\Transition;
use Mockery;

class StateMachineTest extends TestCase
{
    /**
     * @var State[]
     */
    protected $states;

    /**
     * @var State
     */
    protected $initial;

    /**
     * @var State
     */
    protected $final;

    /**
     * @var Transition[]
     */
    protected $transitions;

    /**
     * @var Transition
     */
    protected $approve;

    /**
     * @var \Illuminate\Contracts\Events\Dispatcher|Mockery\LegacyMockInterface|Mockery\MockInterface
     */
    protected $dispatcher;

    /**
     * @var StateMachine
     */
    protected $stateMachine;

    /**
     * @inheritdoc
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->transitions = [
            $this->approve = new Transition('approve', ['submitted'], 'approved')
        ];

        $this->initial = new State('submitted', 'initial', [$this->approve]);
        $this->final = new State('approved', 'final', []);

        $this->states = [
            'submitted' => $this->initial,
            'approved' => $this->final
        ];

        $this->dispatcher = Mockery::spy('Illuminate\Contracts\Events\Dispatcher');

        $this->stateMachine = new StateMachine(
            $this->getStatefulInstance(), $this->states, $this->transitions, $this->dispatcher
        );
    }

    /**
     * @test
     */
    public function it_can_determine_the_current_state_from_the_given_stateful_instance()
    {
        $this->assertEquals($this->stateMachine->state(), $this->initial);
    }

    /**
     * @test
     */
    public function it_can_get_the_given_stateful_instance()
    {
        $this->assertInstanceOf(StatefulInterface::class, $this->stateMachine->getStatefulInstance());
    }

    /**
     * @test
     */
    public function it_can_get_the_given_states()
    {
        $this->assertContainsOnlyInstancesOf(StateInterface::class, $this->stateMachine->getStates());
    }

    /**
     * @test
     */
    public function it_can_get_a_state_object_by_name()
    {
        $state = $this->stateMachine->getState('approved');
        $this->assertInstanceOf(StateInterface::class, $state);
    }

    /**
     * @test
     */
    public function it_will_throw_the_correct_exception_when_it_cannot_find_a_state_object_by_name()
    {
        $this->expectException(StateNotFoundException::class);
        $this->expectExceptionMessage('Unable to find state `foo`');
        $this->stateMachine->getState('foo');
    }

    /**
     * @test
     */
    public function it_can_get_the_given_transitions()
    {
        $this->assertContainsOnlyInstancesOf(TransitionInterface::class, $this->stateMachine->getTransitions());
    }

    /**
     * @test
     */
    public function it_will_throw_the_correct_exception_when_it_cannot_find_a_transition_object_by_name()
    {
        $this->expectException(TransitionNotFoundException::class);
        $this->expectExceptionMessage('Unable to find transition `foo`');
        $this->stateMachine->getTransition('foo');
    }

    /**
     * @test
     */
    public function it_can_get_a_transition_object_by_name()
    {
        $transition = $this->stateMachine->getTransition('approve');
        $this->assertInstanceOf(TransitionInterface::class, $transition);
    }

    /**
     * @test
     */
    public function it_can_check_if_a_state_can_be_transitioned_to_another()
    {
        $this->assertTrue($this->stateMachine->can('approve'));
    }

    /**
     * @test
     */
    public function it_will_fail_if_the_transition_has_not_been_registered_with_the_state()
    {
        $this->assertFalse($this->stateMachine->can('foo'));
    }

    /**
     * @test
     */
    public function it_will_fail_if_the_transition_guard_is_falsey()
    {
        $transitions = [
            'approve' => $approve = new Transition('approve', ['submitted'], 'approved', function () {
                return false;
            })
        ];

        $states = [
            'submitted' => new State('submitted', 'initial', [$approve]),
            'approved' => new State('approved', 'final', [])
        ];

        $stateMachine = $this->getStateMachine($states, $transitions);

        $this->assertFalse($stateMachine->can('approve'));
    }

    /**
     * @test
     */
    public function it_can_transition_from_one_state_to_another()
    {
        $transitions = [
            'approve' => $approve = new Transition('approve', ['submitted'], 'approved')
        ];

        $states = [
            'submitted' => new State('submitted', 'initial', [$approve]),
            'approved' => new State('approved', 'final', [])
        ];

        $dispatcher = Mockery::mock('Illuminate\Contracts\Events\Dispatcher');
        $dispatcher->shouldReceive('dispatch')
                   ->once()
                   ->withArgs(function ($name, $context) {
                       $this->assertEquals(
                           $name,
                           'state-machine.pre-transition.ignite-statemachine-tests-stubs-stateful'
                       );
                       $this->assertInstanceOf(StatefulInterface::class, $context['stateful']);
                       $this->assertInstanceOf(StateInterface::class, $context['state']);
                       $this->assertInstanceOf(TransitionInterface::class, $context['transition']);
                       $this->assertEquals('bar', $context['foo']);
                       return true;
                   });

        $dispatcher->shouldReceive('dispatch')
                   ->withArgs(function ($name, $context) {
                       $this->assertEquals(
                           $name,
                           'state-machine.post-transition.ignite-statemachine-tests-stubs-stateful'
                       );
                       $this->assertInstanceOf(StatefulInterface::class, $context['stateful']);
                       $this->assertInstanceOf(StateInterface::class, $context['state']);
                       $this->assertInstanceOf(TransitionInterface::class, $context['transition']);
                       $this->assertEquals('bar', $context['foo']);
                       return true;
                   });

        $stateMachine = new StateMachine(
            $this->getStatefulInstance(), $states, $transitions, $dispatcher
        );

        $stateMachine->apply('approve', ['foo' => 'bar']);

        $this->assertEquals('approved', $stateMachine->state()->name());
        $this->assertEquals('approved', $stateMachine->getStatefulInstance()->getState());
    }

    /**
     * @test
     */
    public function it_can_dispatch_an_arbitrary_state_machine_event()
    {
        $transitions = [
            'approve' => $approve = new Transition('approve', ['submitted'], 'approved')
        ];

        $states = [
            'submitted' => new State('submitted', 'initial', [$approve]),
            'approved' => new State('approved', 'final', [])
        ];

        $dispatcher = Mockery::mock('Illuminate\Contracts\Events\Dispatcher');
        $dispatcher->shouldReceive('dispatch')
                   ->once()
                   ->withArgs(function ($name, $context) {
                       $this->assertEquals(
                           $name,
                           'state-machine.foo-bar.ignite-statemachine-tests-stubs-stateful'
                       );
                       $this->assertInstanceOf(StatefulInterface::class, $context['stateful']);
                       $this->assertInstanceOf(StateInterface::class, $context['state']);
                       $this->assertInstanceOf(TransitionInterface::class, $context['transition']);
                       $this->assertEquals('bar', $context['foo']);
                       return true;
                   });

        $stateMachine = new StateMachine(
            $this->getStatefulInstance(), $states, $transitions, $dispatcher
        );

        $stateMachine->dispatchTransitionEvent('foo-bar', $approve, ['foo' => 'bar']);
    }
}
