<?php

namespace Ignite\StateMachine;

use Exception;
use Ignite\StateMachine\Contracts\StatefulInterface;
use Ignite\StateMachine\Contracts\StateInterface;
use Ignite\StateMachine\Contracts\StateMachineInterface;
use Ignite\StateMachine\Contracts\TransitionInterface;
use Ignite\StateMachine\Exceptions\StateNotFoundException;
use Ignite\StateMachine\Exceptions\TransitionNotAllowedException;
use Ignite\StateMachine\Exceptions\TransitionNotFoundException;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Support\Str;

class StateMachine implements StateMachineInterface
{
    /**
     * @var StatefulInterface
     */
    protected $stateful;

    /**
     * @var array
     */
    protected $transitions;

    /**
     * @var array
     */
    protected $states;

    /**
     * @var StateInterface
     */
    protected $state;

    /**
     * @var Dispatcher
     */
    protected $dispatcher;

    /**
     * StateMachine constructor.
     *
     * @param StatefulInterface $stateful
     * @param array $states
     * @param array $transitions
     * @param Dispatcher $dispatcher
     */
    public function __construct(
        StatefulInterface $stateful,
        array $states,
        array $transitions,
        Dispatcher $dispatcher
    ) {
        $this->stateful = $stateful;
        $this->states = $this->setKeysFor($states);
        $this->transitions = $this->setKeysFor($transitions);
        $this->dispatcher = $dispatcher;
        $this->state = $this->setCurrentStateOrFallback();
    }

    /**
     * @inheritdoc
     */
    public function getStatefulInstance()
    {
        return $this->stateful;
    }

    /**
     * @inheritdoc
     */
    public function state()
    {
        return $this->state;
    }

    /**
     * @inheritdoc
     */
    public function can($name, $context = [])
    {
        try {
            $transition = $this->getTransition($name);
        } catch (Exception $e) {
            return false;
        }

        $callable = $transition->guard();

        if (! is_null($callable) && false === $callable($this)) {
            return false;
        }

        if (! array_key_exists($transition->name(), $this->state->transitions())) {
            return false;
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function apply($name, $context = [])
    {
        $transition = $this->getTransition($name);

        if (! $this->can($name, $context)) {
            throw new TransitionNotAllowedException($transition, $this->state, $this->stateful);
        }

        $state = $this->getState($transition->state());

        $this->dispatchPreTransitionEvent($transition, $context);

        $value = $transition->process($this);
        $this->stateful->setState($transition->state());
        $this->state = $state;

        $this->dispatchPostTransitionEvent($transition, $context);

        return $value;
    }

    /**
     * @inheritdoc
     */
    public function getTransition($name)
    {
        if (! isset($this->transitions[$name])) {
            throw new TransitionNotFoundException($name, $this->transitions, $this->stateful);
        }

        return $this->transitions[$name];
    }

    /**
     * @inheritdoc
     */
    public function getTransitions()
    {
        return $this->transitions;
    }

    /**
     * @inheritdoc
     */
    public function getState($name)
    {
        if (! isset($this->states[$name])) {
            throw new StateNotFoundException($name, $this->states, $this->stateful);
        }

        return $this->states[$name];
    }

    /**
     * @inheritdoc
     */
    public function getStates()
    {
        return $this->states;
    }

    /**
     * @inheritdoc
     */
    public function getInitialState()
    {
        $matches = array_filter($this->states, function ($state) {
            return StateInterface::INITIAL === $state->type();
        });

        if (empty($matches)) {
            return null;
        }

        return array_shift($matches);
    }

    /**
     * @inheritdoc
     */
    public function dispatchTransitionEvent(
        $name,
        TransitionInterface $transition,
        array $context = [],
        StatefulInterface $stateful = null
    ) {
        return $this->dispatcher->dispatch($this->eventName($name), array_merge([
            'state' => $this->state,
            'transition' => $transition,
            'stateful' => $stateful ?? $this->stateful
        ], $context));
    }

    /**
     * Set the object, ensuring that any numeric keys are reset to the object name.
     *
     * @param array $input
     *
     * @return array
     */
    protected function setKeysFor($input)
    {
        $output = [];

        foreach ($input as $key => $object) {
            if (is_numeric($key)) {
                $output[$object->name()] = $object;
            } else {
                $output[$key] = $object;
            }
        }

        return $output;
    }

    /**
     * Set the current state or fallback to the initial state.
     *
     * @return string
     */
    protected function setCurrentStateOrFallback()
    {
        $matches = array_filter($this->states, function ($state) {
            return $state->name() === $this->stateful->getState();
        });

        if (empty($matches)) {
            return $this->getInitialState();
        }

        return array_shift($matches);
    }

    /**
     * Dispatch a transition event with the context of the state machine before the given transition is applied.
     *
     * @param TransitionInterface $transition
     * @param array $context
     *
     * @return array|null
     */
    protected function dispatchPreTransitionEvent(TransitionInterface $transition, array $context = [])
    {
        return $this->dispatchTransitionEvent('pre-transition', $transition, $context);
    }

    /**
     * Dispatch a transition event with the context of the state machine after the given transition is applied.
     *
     * @param TransitionInterface $transition
     * @param array $context
     *
     * @return array|null
     */
    protected function dispatchPostTransitionEvent(TransitionInterface $transition, array $context = [])
    {
        return $this->dispatchTransitionEvent('post-transition', $transition, $context);
    }

    /**
     * Format the event name to be dispatched.
     *
     * @param string $name
     *
     * @return string
     */
    protected function eventName($name)
    {
        return sprintf(
            'state-machine.%s.%s',
            strtolower($name),
            Str::slug(str_replace('\\', '-', get_class($this->stateful)))
        );
    }
}
