<?php

namespace Ignite\Extensions\Monolog\Reader;

use Ignite\Extensions\Monolog\Parser\LogParserInterface;
use RuntimeException;
use SplFileObject;

class LogReader extends AbstractReader implements \Iterator, \ArrayAccess, \Countable
{
    /**
     * @var SplFileObject
     */
    protected $file;

    /**
     * @var int
     */
    protected $lineCount;

    /**
     * @var LogParserInterface
     */
    protected $parser;

    /**
     * Create a new instance of the log reader.
     *
     * @param string $file
     * @param string|null $defaultPatternPattern
     * @param LogParserInterface|null $parser
     */
    public function __construct(string $file, $defaultPatternPattern = null, $parser = null)
    {
        parent::__construct($defaultPatternPattern);

        $this->file = new SplFileObject($file, 'r');
        $this->lineCount = $this->getLineCount();
        $this->parser = $parser ?? $this->getDefaultParser();
    }

    /**
     * Get the number of lines in the log file.
     *
     * @return int
     */
    private function getLineCount()
    {
        $i = 0;

        while (! $this->file->eof()) {
            $this->file->current();
            $this->file->next();
            $i++;
        }

        return $i;
    }

	/**
	 * @param LogParserInterface $parser
	 * @return $this
	 */
	public function setParser(LogParserInterface $parser)
	{
		$this->parser = $parser;

		return $this;
	}

    /**
     * {@inheritdoc}
     */
    public function offsetExists($offset)
    {
        return $offset < $this->lineCount;
    }

    /**
     * {@inheritdoc}
     */
    public function offsetGet($offset)
    {
        $key = $this->file->key();
        $this->file->seek($offset);
        $log = $this->current();
        $this->file->seek($key);
        $this->file->current();

        return $log;
    }

    /**
     * {@inheritdoc}
     */
    public function offsetSet($offset, $value)
    {
        throw new RuntimeException("LogReader is read-only.");
    }

    /**
     * {@inheritdoc}
     */
    public function offsetUnset($offset)
    {
        throw new RuntimeException("LogReader is read-only.");
    }

    /**
     * {@inheritdoc}
     */
    public function rewind()
    {
        $this->file->rewind();
    }

    /**
     * {@inheritdoc}
     */
    public function next()
    {
        $this->file->next();
    }

    /**
     * {@inheritdoc}
     */
    public function current()
    {
        return $this->parser->parse($this->file->current());
    }

    /**
     * {@inheritdoc}
     */
    public function key()
    {
        return $this->file->key();
    }

    /**
     * {@inheritdoc}
     */
    public function valid()
    {
        return $this->file->valid();
    }

    /**
     * {@inheritdoc}
     */
    public function count()
    {
        return $this->lineCount;
    }
}
