<?php

namespace Ignite\Extensions\Monolog\Reader;

use Ignite\Extensions\Monolog\Parser\LogParserInterface;
use RuntimeException;
use SplFileObject;

class LogReader extends AbstractReader implements \Iterator, \ArrayAccess, \Countable
{
    /**
     * @var SplFileObject
     */
    protected $file;

    /**
     * @var int
     */
    protected $lineCount;

    /**
     * @var LogParserInterface
     */
    protected $parser;

    /**
     * Create a new instance of the log reader.
     *
     * @param string $file
     * @param string|null $defaultPatternPattern
     * @param LogParserInterface|null $parser
     */
    public function __construct(string $file, $defaultPatternPattern = null, $parser = null)
    {
        parent::__construct($defaultPatternPattern);

        $this->file = new SplFileObject($file, 'r');
        $this->lineCount = $this->getLineCount();
        $this->parser = $parser ?? $this->getDefaultParser();
    }

    /**
     * Get the number of lines in the log file.
     *
     * @return int
     */
    private function getLineCount()
    {
        $i = 0;

        while (! $this->file->eof()) {
            $this->file->current();
            $this->file->next();
            $i++;
        }

        return $i;
    }

	/**
	 * @param LogParserInterface $parser
	 * @return $this
	 */
	public function setParser(LogParserInterface $parser)
	{
		$this->parser = $parser;

		return $this;
	}

    /**
     * {@inheritdoc}
     */
    public function offsetExists($offset): bool
    {
        return $offset < $this->lineCount;
    }

    /**
     * {@inheritdoc}
     */
    public function offsetGet($offset): mixed
    {
        $key = $this->file->key();
        $this->file->seek($offset);
        $log = $this->current();
        $this->file->seek($key);
        $this->file->current();

        return $log;
    }

    /**
     * {@inheritdoc}
     */
    public function offsetSet($offset, $value): void
    {
        throw new RuntimeException("LogReader is read-only.");
    }

    /**
     * {@inheritdoc}
     */
    public function offsetUnset($offset): void
    {
        throw new RuntimeException("LogReader is read-only.");
    }

    /**
     * {@inheritdoc}
     */
    public function rewind(): void
    {
        $this->file->rewind();
    }

    /**
     * {@inheritdoc}
     */
    public function next(): void
    {
        $this->file->next();
    }

    /**
     * {@inheritdoc}
     */
    public function current(): mixed
    {
        return $this->parser->parse($this->file->current());
    }

    /**
     * {@inheritdoc}
     */
    public function key(): mixed
    {
        return $this->file->key();
    }

    /**
     * {@inheritdoc}
     */
    public function valid(): bool
    {
        return $this->file->valid();
    }

    /**
     * {@inheritdoc}
     */
    public function count(): int
    {
        return $this->lineCount;
    }
}
