<?php

namespace Ignite\HelixCatalog\Transformers;

use Ignite\HelixCatalog\Contracts\PayloadTransformerInterface;
use Ignite\HelixCatalog\Entities\User;
use Ignite\Vendor\Helix\Laravel\Services\Token;
use Illuminate\Config\Repository;

class PayloadTransformer implements PayloadTransformerInterface
{
    /**
     * @var Token
     */
    protected $tokenService;

    /**
     * @var Repository
     */
    protected $config;

    /**
     * PayloadTransformer constructor.
     *
     * @param Token $tokenService
     * @param Repository $config
     */
    public function __construct(Token $tokenService, Repository $config)
    {
        $this->tokenService = $tokenService;
        $this->config = $config;
    }

    /**
     * Transform the user for use with Helix.
     *
     * @param User $user
     *
     * @return array
     */
    public function transform($user)
    {
        if (! $user->relationLoaded('participant')) {
            $user->load('participant');
        }

        $payload = [
            'clientUserId' => $user->getKey(),
            'catalogId' => $this->config->get('helix.catalog'),
            'accessToken' => (string) $this->getAccessToken($user),
            'firstName' => (string) $user->first,
            'lastName' => (string) $user->last,
            'primaryLanguage' => (string) $this->getPrimaryLanguage($user),
            'points' => (int) $user->points,
            'emailAddress' => (string) $user->email,
            'phoneNumber1' => (string) optional($user->participant)->phone1 ?? '',
            'shipAddress1' => (string) optional($user->participant)->address_1 ?? '',
            'shipAddress2' => (string) optional($user->participant)->address_2 ?? '',
            'shipCity' => (string) optional($user->participant)->city ?? '',
            'shipState' => (string) optional($user->participant)->state ?? '',
            'shipPostalCode' => (string) optional($user->participant)->postal ?? '',
            'shipCountry' => (string) strtoupper(optional($user->participant)->country) ?? '',
        ];

        if (! empty($user->helix_user_id)) {
            $payload['helixUserId'] = $user->helix_user_id;
        }

        return $payload;
    }

    /**
     * Get the encrypted access token for the given user.
     *
     * @param User $user
     *
     * @return string
     */
    protected function getAccessToken(User $user)
    {
        return $this->tokenService->encrypt($this->config->get('helix.token.catalog'), $user->getKey());
    }

    /**
     * Get the user's configured primary language.
     *
     * @param User $user
     * @param string $default
     *
     * @return bool|string
     */
    protected function getPrimaryLanguage(User $user, $default = 'en')
    {
        if (empty($user->participant) || empty($user->participant->language_preference)) {
            return $default;
        }

        $code = strtolower(substr($user->participant->language_preference, 0, 2));

        return $this->config->get("helix-catalog.languages.{$code}", $default);
    }
}
