<?php

namespace Ignite\HelixCatalog\Listeners;

use Exception;
use Ignite\Catalog\Entities\Order;
use Ignite\HelixCatalog\Emails\OrderShipped;
use Ignite\HelixCatalog\Entities\Shipment;
use Ignite\Vendor\Helix\Laravel\Resources\Webhook;
use Ignite\Vendor\Helix\Laravel\Events\WebhookResourceCreated;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Mail;
use Throwable;

class UpdateShippingOrderStatus
{
    /**
     * Handle the event.
     *
     * @param WebhookResourceCreated $event
     *
     * @return void
     * @throws Throwable
     */
    public function handle(WebhookResourceCreated $event)
    {
        if (! config('helix.enabled')) {
            return;
        }

        $webhook = $event->getWebhook();

        try {
            $this->guardAgainstMissingOrderNumber($webhook);

            $order = $this->getOrder($webhook);

            $shipment = $order->getConnection()->transaction(function () use ($order, $webhook) {
                foreach ($order->items as $item) {
                    $item->forceFill([
                        'shipped' => 1,
                        'shipped_quantity' => $item->quantity,
                        'shipped_at' => $webhook->payload['shippingDate'],
                    ])->save();
                }

                $shipment = Shipment::updateOrCreate(['order_id' => $order->getKey()], [
                    'order_id' => $order->getKey(),
                    'order_number' => $webhook->payload['orderNumber'],
                    'carrier' => $webhook->payload['carrier'],
                    'tracking_number' => $webhook->payload['trackingNumber'],
                    'ship_date' => $webhook->payload['shippingDate'],
                ]);

                $webhook->processed();

                return $shipment;
            });

            $order = $order->refresh();

            Mail::to([$order->participant->fullName() => $order->ship_email])
                ->locale($order->participant->language_preference)
                ->queue(new OrderShipped($order, $shipment));
        } catch (Throwable $exception) {
            logger()->error($exception->getMessage(), compact('webhook', 'exception'));
            if (config('helix-catalog.webhook.exceptions')) {
                throw $exception;
            }
        }
    }

    /**
     * Ensure we have an order number.
     *
     * @param Webhook $webhook
     *
     * @throws Exception
     */
    private function guardAgainstMissingOrderNumber(Webhook $webhook)
    {
        if (! isset($webhook->payload['orderNumber']) || empty($webhook->payload['orderNumber'])) {
            throw new Exception("Missing order number for Webhook: ID - {$webhook->getKey()}");
        }
    }

    /**
     * Get the related order.
     *
     * @param Webhook $webhook
     *
     * @return Order
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException
     */
    private function getOrder(Webhook $webhook)
    {
        return Order::query()
            ->with('items', 'user', 'participant')
            ->byNumber($webhook->payload['orderNumber'])
            ->firstOrFail();
    }
}
