<?php

namespace Ignite\HelixCatalog\Listeners;

use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Catalog\Entities\Vendor;
use Ignite\Catalog\Events\OrderSubmitted;
use Ignite\Core\Entities\Transaction;
use Ignite\Vendor\Helix\Laravel\Events\OrderResourceCreated;
use Ignite\Vendor\Helix\Laravel\Resources\Order as HelixOrder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;

class StoreOrderDetails
{
    /**
     * Handle the event.
     *
     * @param OrderResourceCreated $event
     *
     * @return void
     * @throws \Throwable
     */
    public function handle(OrderResourceCreated $event)
    {
        if (! config('helix.enabled')) {
            return;
        }

        $helixOrder = $event->getOrder();

        if (! $helixOrder->isProcessed()) {
            try {
                $helixOrder->getConnection()->transaction(function () use ($helixOrder) {
                    $order = $this->transformOrder($helixOrder);
                    $helixOrder->processed();
                    return $order;
                });
            } catch (\Exception $exception) {
                logger()->error($exception->getMessage(), compact('helixOrder', 'exception'));
                if (config('helix-catalog.webhook.exceptions')) {
                    throw $exception;
                }
            }
        }
    }

    /**
     * Tranform the helix order into an Ignite order.
     *
     * @param HelixOrder $helixOrder
     *
     * @return mixed
     */
    protected function transformOrder(HelixOrder $helixOrder)
    {
        $vendor = $this->findHelixVendor();
        $catalog = $this->findHelixCatalog();

        $order = $this->createOrder($helixOrder);

        foreach ($helixOrder->products as $product) {
            $item = $this->createProductItem($product, $helixOrder, $order, $catalog, $vendor);
            $orderItem = $this->createOrderItem($product, $helixOrder, $order, $item, $vendor);
        }

        $this->createTransaction($order, $helixOrder);

        event(new OrderSubmitted($order));

        return $order;
    }

    /**
     * The helix vendor record.
     *
     * @return Vendor|Model
     */
    protected function findHelixVendor()
    {
        $names = ['helix-iframe'];
        return Vendor::query()->whereIn('name', $names)->firstOrFail();
    }

    /**
     * The helix catalog record.
     *
     * @return Catalog|Model
     */
    protected function findHelixCatalog()
    {
        $names = ['helix-iframe-catalog', 'helix-catalog'];
        return Catalog::query()->whereIn('code', $names)->firstOrFail();
    }

    /**
     * The data required to create the order.
     *
     * @param HelixOrder $helixOrder
     *
     * @return array
     */
    protected function getOrderData(HelixOrder $helixOrder)
    {
        return [
            'user_id' => $helixOrder->user_id,
            'quantity' => $helixOrder->qty,
            'points' => $helixOrder->points,
            'ship_name' => "{$helixOrder->first} {$helixOrder->last}",
            'ship_email' => $helixOrder->ship_email,
            'ship_phone' => $helixOrder->ship_phone,
            'ship_address_1' => $helixOrder->ship_address_1,
            'ship_address_2' => $helixOrder->ship_address_2,
            'ship_address_3' => '',
            'ship_city' => $helixOrder->ship_city,
            'ship_state' => $helixOrder->ship_state,
            'ship_postal' => $helixOrder->ship_postcode,
            'ship_country' => $helixOrder->ship_country,
            'processed' => 1,
            'processed_at' => now(),
        ];
    }

    /**
     * Create the Ignite order from Helix order data.
     *
     * @param HelixOrder $helixOrder
     *
     * @return mixed
     */
    protected function createOrder(HelixOrder $helixOrder)
    {
        $orderData = $this->getOrderData($helixOrder);

        $orderItemMatch = OrderItem::query()
            ->whereJsonContains('vendor_meta->number', $helixOrder->order_number)
            ->first();

        if ($orderItemMatch) {
            $order = Order::query()->whereKey($orderItemMatch->catalog_order_id)->first();
        } else {
            $order = new Order(['number' => 'PROCESSING']);
        }

        $order->forceFill($orderData)->save();

        return $order->refresh();
    }

    /**
     * Create the product item from the available order item data.
     *
     * @param array $product
     * @param HelixOrder $helixOrder
     * @param Order $order
     * @param Catalog $catalog
     * @param Vendor $vendor
     *
     * @return Item|Model
     */
    protected function createProductItem(
        array $product,
        HelixOrder $helixOrder,
        Order $order,
        Catalog $catalog,
        Vendor $vendor
    ) {
        return Item::query()->updateOrCreate([
            'catalog_id' => $catalog->getKey(),
            'catalog_vendor_id' => $vendor->getKey(),
            'sku' => $product['id'],
        ], [
            'catalog_id' => $catalog->getKey(),
            'catalog_vendor_id' => $vendor->getKey(),
            'code' => $product['id'],
            'sku' => $product['id'],
            'type' => 'simple',
            'class' => 'merchandise',
            'name' => $product['name'],
            'description' => $product['shortDescription'],
            'manufacturer' => $product['brandName'],
            'image' => $product['imageUrl'],
            'msrp' => $product['msrp'],
            'cost' => $product['costPerItem'],
            'price' => $product['costPerItem'],
            'price_markup' => 0,
            'price_margin' => 0,
            'point_value' => 1,
            'points' => $product['pointPerItem'],
            'points_min' => 0,
            'points_max' => 0,
            'visibility' => 1,
            'active' => 1,
            'vendor_active' => 1,
            'vendor_meta' => [
                'model_number' => $product['modelNumber'],
                'category_name' => $product['categoryName'],
                'is_featured' => $product['isFeatured'],
                'is_special' => $product['isSpecial'],
            ]
        ]);
    }

    /**
     * Create the order item from the available order data.
     *
     * @param array $product
     * @param Order $order
     * @param Item $item
     * @param Vendor $vendor
     * @param HelixOrder $helixOrder
     *
     * @return \Illuminate\Database\Eloquent\Builder|Model
     */
    protected function createOrderItem(array $product, HelixOrder $helixOrder, Order $order, Item $item, Vendor $vendor)
    {
        return OrderItem::query()->updateOrCreate([
            'catalog_order_id' => $order->getKey(),
            'sku' => $product['id'],
        ], [
            'catalog_order_id' => $order->getKey(),
            'catalog_item_id' => $item->getKey(),
            'catalog_vendor_id' => $vendor->getKey(),
            'vendor_order_number' => $helixOrder->order_number,
            'sku' => $product['id'],
            'name' => $product['name'],
            'type' => $product['productType'],
            'class' => 'merchandise',
            'category' => $product['categoryName'],
            'quantity' => $product['quantity'],
            'cost' => $product['costPerItem'],
            'price' => $product['msrp'],
            'point_value' => 1,
            'points' => $product['pointPerItem'],
            'processed' => 1,
            'processed_at' => now(),
            'processed_quantity' => $product['quantity'],
            'vendor_meta' => [
                'number' => $helixOrder->order_number,
                'attempts' => [
                    'helix-1' => [
                        'response' => [
                            'success' => true,
                            'isCompleted' => true,
                        ],
                        'item' => [
                            'order_number' => $helixOrder->order_number,
                            'client_user_id' => $helixOrder->user_id,
                        ]
                    ]
                ]
            ]
        ]);
    }

    /**
     * Create the transaction entry to redeem the points.
     *
     * @param Order $order
     * @param HelixOrder $helixOrder
     *
     * @return Transaction
     */
    protected function createTransaction(Order $order, HelixOrder $helixOrder)
    {
        return Transaction::updateOrCreate([
            'user_id' => $helixOrder->user_id,
            'related_id' => $order->getKey(),
            'related_name' => 'ORDER',
            'related_type' => Order::class,
            'description' => $order->number,
            'type' => Transaction::REDEEMED,
        ], [
            'user_id' => $helixOrder->user_id,
            'related_id' => $order->getKey(),
            'related_name' => 'ORDER',
            'related_type' => Order::class,
            'description' => $order->number,
            'type' => Transaction::REDEEMED,
            'value' => - (float) $helixOrder->points,
            'tax_date' => now(),
            'transaction_date' => now(),
        ]);
    }
}
