<?php

namespace Ignite\Vendor\Helix\Models;

use Countable;

abstract class Collection implements Countable
{
    /**
     * The collection items.
     *
     * @var array
     */
    private $items = [];

    /**
     * Collection constructor.
     *
     * @param array $items
     */
    public function __construct(array $items = [])
    {
        $this->addMany($items);
    }

    /**
     * Add an item to the collection.
     *
     * @param  Model $item
     * @param  null|string|int $key
     * @return $this
     */
    protected function add(Model $item, $key = null)
    {
        if (is_null($key)) {
            $this->items[] = $item;
        } else {
            $this->items[$key] = $item;
        }

        return $this;
    }

    /**
     * Add many items via an array.
     *
     * @param array $items
     */
    protected function addMany(array $items = [])
    {
        foreach ($items as $item) {
            $this->add($item);
        }
    }

    /**
     * Remove an item to the collection.
     *
     * @param  string|int $key
     * @return $this
     */
    protected function remove($key)
    {
        unset($this->items[$key]);

        return $this;
    }

    /**
     * Get the item at the given index.
     *
     * @param  string|int $key
     * @return Model
     */
    protected function get($key)
    {
        return $this->items[$key];
    }

    /**
     * The number of key-value pairs in the data array.
     *
     * @return int
     */
    public function count()
    {
        return count($this->items);
    }

    /**
     * Cast the collection to an array of arrays.
     *
     * @return array
     */
    public function toArray()
    {
        return array_values(array_map(function (Model $item) {
            return $item->toArray();
        }, $this->items));
    }

    /**
     * Cast the collection to JSON.
     *
     * @return false|string
     */
    public function toJson()
    {
        return json_encode($this->toArray(), JSON_NUMERIC_CHECK);
    }
}
