<?php

namespace Ignite\Vendor\Helix\Models;

use Countable;

abstract class Model implements Countable
{
    /**
     * The model data.
     *
     * @var array
     */
    protected $data = [];

    /**
     * OrderItem constructor.
     *
     * @param array $data
     */
    public function __construct(array $data)
    {
        $this->fill($data);
    }

    /**
     * Fill the model with the given data.
     *
     * @param  array $data
     * @return array
     */
    protected function fill(array $data)
    {
        foreach ($data as $key => $value) {
            $value = trim($value);
            $method = $this->camelize('set_' . $key);
            if (method_exists($this, $method)) {
                call_user_func([$this, $method], $value);
            } else {
                $this->data[$key] = $value;
            }
        }

        return $this->data;
    }

    /**
     * Transform a snake_case string into a camelCase string.
     *
     * @param  string $key
     * @return string
     */
    protected function camelize($key)
    {
        return lcfirst(str_replace('_', '', ucwords($key, '_')));
    }

    /**
     * Transform a camelCase string into a snake_case string.
     *
     * @param  string $key
     * @return string
     */
    protected function decamelize($key)
    {
        return ltrim(strtolower(preg_replace('/[A-Z]([A-Z](?![a-z]))*/', '_$0', $key)), '_');
    }

    /**
     * Call a magic method on the object.
     *
     * @param  string $name
     * @param  array $arguments
     * @return mixed
     * @throws \Exception
     */
    public function __call($name, $arguments)
    {
        $action = strtolower(substr($name, 0, 3));
        $key = substr($name, 3);

        array_unshift($arguments);

        if (! in_array($action, ['set', 'get'])) {
            throw new \Exception("Unknown action $action");
        }

        return call_user_func_array([$this, $action], array_merge([$key], $arguments));
    }

    /**
     * Get a data value by key.
     *
     * @param  string $key
     * @param  mixed  $default
     * @return mixed
     */
    protected function get($key, $default = null)
    {
        $camel = $this->camelize($key);
        $snake = $this->decamelize($key);

        // Check for key in camel case: firstName
        if (array_key_exists($camel, $this->data)) {
            return $this->data[$camel];
        }

        // Check for key in studly case: FirstName
        if (array_key_exists($search = ucfirst($camel), $this->data)) {
            return $this->data[$search];
        }

        // Check for key in snake case: first_name
        if (array_key_exists($snake, $this->data)) {
            return $this->data[$snake];
        }

        return $default;
    }

    /**
     * Set a data value by key.
     *
     * @param  string $key
     * @param  mixed  $value
     * @return mixed
     */
    protected function set($key, $value)
    {
        $this->data[$key] = $value;

        return $this;
    }

    /**
     * Cast the object to an array.
     *
     * @return array
     */
    public function toArray()
    {
        return $this->data;
    }

    /**
     * Cast the object to JSON.
     *
     * @return false|string
     */
    public function toJson()
    {
        return json_encode($this->toArray(), JSON_NUMERIC_CHECK);
    }

    /**
     * The number of key-value pairs in the data array.
     *
     * @return int
     */
    public function count()
    {
        return count($this->data);
    }
}
