<?php

namespace Ignite\Vendor\Helix\Tests;

use Ignite\Vendor\Helix\Adapters\Guzzle;
use Ignite\Vendor\Helix\Client;
use Ignite\Vendor\Helix\Authentication;
use Ignite\Vendor\Helix\Models\Request\OrderItem;
use Ignite\Vendor\Helix\Transformers\Json;

class ClientTest extends TestCase
{
    /**
     * @var Authentication
     */
    private $auth;

    /**
     * @var Guzzle
     */
    private $adapter;

    /**
     * @var Client
     */
    private $client;

    /**
     * Set up for each isolated test.
     */
    public function setUp()
    {
        $this->auth = new Authentication($this->username, $this->password, $this->endpoint);
        $this->adapter = new Guzzle($this->auth);
        $this->client = new Client($this->adapter);
    }

    /**
     * @test
     */
    public function it_can_signup_a_new_user()
    {
        $this->markTestSkipped('The API endpoint we have does not support SSO.');
        $data = [
            "catalogId" => "A14E715F-CAF0-4749-B569-58059C9DD490",
            "accessToken" => "2F9C54FC8D7A826B7852DC0BED757113F44E7583EDF65DE4FB253A23B79F1862",
            "clientUserId" => "12345",
            "firstName" => "Hamish",
            "lastName" => "McPants",
            "PrimaryLanguage" => "en",
            "points" => "500",
            "homeAddress1" => "144 Dundeee Street",
            "homeAddress2" => "Apt 29",
            "homeCity" => "Northbook",
            "homeState" => "IL",
            "homePostalCode" => "60062",
            "homeCountry" => "US",
            "shipAddress1" => "44 Dundeee Street",
            "shipAddress2" => "Apt 29",
            "shipCity" => "Northbrook",
            "shipState" => "IL",
            "shipPostalCode" => "60062",
            "shipCountry" => "US",
            "phoneNumber1" => "9620245835",
            "phoneNumber2" => "9620245835",
            "emailAddress" => "hamish.mcpants@test.com",
            "customField1" => "CF041416-7EAD-46C4-C411-B4CF82133DE3",
            "customField2" => "123",
            "customField3" => "",
            "Yos" => "5"
        ];

        $auth = new Authentication($this->username, $this->password, $this->endpoint);
        $adapter = new Guzzle($auth);
        $client = new Client($adapter);
        $response = $client->signUp($data);

        var_dump($response->getBody()->getContents());

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);

        var_dump($json);
    }

    /**
     * @test
     */
    public function it_can_fetch_a_list_of_brands()
    {
        $response = $this->client->getBrandList();

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('data', $json);
        $this->assertArrayHasKey('brandId', array_shift($json['data']));
    }

    /**
     * @test
     */
    public function it_can_fetch_a_list_of_main_categories()
    {
        $response = $this->client->getCategoryList();

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertArrayHasKey('categoryId', array_shift($json['data']));
    }

    /**
     * @test
     */
    public function it_can_fetch_a_list_of_sub_categories()
    {
        $response = $this->client->getSubCategoryList(2);

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertArrayHasKey('categoryId', array_shift($json['data']));
    }

    /**
     * @test
     */
    public function it_can_fetch_a_list_of_category_products()
    {
        $response = $this->client->getProductList(2);

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertArrayHasKey('productId', array_shift($json['data']));
    }

    /**
     * @test
     */
    public function it_can_fetch_a_list_of_featured_products()
    {
        $response = $this->client->getFeaturedProducts();

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertArrayHasKey('productId', array_shift($json['data']));
    }

    /**
     * @test
     */
    public function it_can_fetch_a_list_of_special_products()
    {
        $response = $this->client->getSpecialProducts();

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertArrayHasKey('productId', array_shift($json['data']));
    }

    /**
     * @test
     */
    public function it_can_fetch_a_list_of_new_products()
    {
        $response = $this->client->getNewProducts();

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertArrayHasKey('productId', array_shift($json['data']));
    }

    /**
     * @test
     */
    public function it_can_fetch_a_list_of_similar_products()
    {
        $response = $this->client->getSimilarProducts(2587);

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertArrayHasKey('productId', array_shift($json['data']));
    }

    /**
     * @test
     */
    public function it_can_fetch_product_details()
    {
        $response = $this->client->getProductDetails(2587);

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertArrayHasKey('productId', array_shift($json['data']));
    }

    /**
     * @test
     */
    public function it_can_fetch_order_details()
    {
        $response = $this->client->getOrderDetails(194459);

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertEquals(194459, array_shift($json['data'])['orderId']);
    }

    /**
     * @test
     */
    public function it_can_cancel_an_order()
    {
        $response = $this->client->cancelOrder(194459);

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertEquals('success', $json['status']);
    }

    /**
     * @test
     */
    public function it_can_place_an_order()
    {
        $billingItems = [
            $this->getBillingData()
        ];
        $orderItems = [
            ['product_id' => 2995, 'qty' => 2],
            ['product_id' => 4267, 'qty' => 2]
        ];

        $response = $this->client->placeOrder($billingItems, $orderItems);

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertEquals('success', $json['status']);
        $this->assertArrayHasKey('COMMANDE_ID', $json);
        $this->assertArrayHasKey('message', $json);
        $this->assertEquals('Order placed successfully', $json['message']);
    }

    /**
     * @test
     */
    public function it_can_place_an_order_for_an_egiftcard()
    {
        $this->markTestSkipped('The API endpoint is not reliable for this request.');

        $billingItems = [
            $this->getBillingData()
        ];
        $orderItems = [
            [
                'product_id' => 683,
                'qty' => 1,
                'ecertificate' => OrderItem::CARD_TYPE_ECERTIFICATE
            ]
        ];

        $response = $this->client->placeOrder($billingItems, $orderItems);

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertEquals('success', $json['status']);
        $this->assertArrayHasKey('COMMANDE_ID', $json);
        $this->assertArrayHasKey('message', $json);
        $this->assertEquals('Order placed successfully', $json['message']);
    }

    /**
     * @test
     */
    public function it_can_place_an_order_for_a_giftcard()
    {
        $billingItems = [
            $this->getBillingData()
        ];
        $orderItems = [
            [
                'product_id' => 683,
                'qty' => 1,
                'ecertificate' => OrderItem::CARD_TYPE_PHYSICAL
            ]
        ];

        $response = $this->client->placeOrder($billingItems, $orderItems);

        $this->assertResponseSuccessful($response);

        $json = (new Json)->transform($response);
        $this->assertArrayHasKey('status', $json);
        $this->assertEquals('success', $json['status']);
        $this->assertArrayHasKey('COMMANDE_ID', $json);
        $this->assertArrayHasKey('message', $json);
        $this->assertEquals('Order placed successfully', $json['message']);
    }
}
