<?php

namespace Ignite\Vendor\Helix\Adapters;

use GuzzleHttp\Client;
use Ignite\Vendor\Helix\AdapterInterface;
use Ignite\Vendor\Helix\Authentication;

final class Guzzle implements AdapterInterface
{
    /**
     * The Guzzle Client instance.
     *
     * @var Client
     */
    private $guzzle;

    /**
     * The authenication instance with username, password and uri.
     *
     * @var Authentication
     */
    private $auth;

    /**
     * The default options.
     *
     * @var array
     */
    private $options = [
        'debug' => false,
        'http_errors' => false,
        'headers' => [
            'Content-Type' => 'application/json',
        ],
    ];

    /**
     * Instantiate a new instance of the Guzzle adapter.
     *
     * @param Authentication $auth
     * @param Client $guzzle
     * @param array $options
     */
    public function __construct(Authentication $auth, Client $guzzle = null, array $options = [])
    {
        $this->auth = $auth;
        $this->setOptions($options);
        $this->guzzle = $guzzle ?: $this->getDefaultInstance();
    }

    /**
     * Set the options on the client.
     *
     * @param array $options
     */
    private function setOptions(array $options = [])
    {
        $this->options = array_merge($this->options, [
            'base_uri' => $this->auth->endpoint(),
            'auth' => [$this->auth->username(), $this->auth->password()],
            'headers' => [
                'Authorization' => 'Bearer ' . $this->token()
            ]
        ], $options);
    }

    /**
     * Create a default configured instance of guzzle.
     *
     * @param  array $options
     * @return Client
     */
    public function getDefaultInstance(array $options = [])
    {
        if (! empty($options)) {
            $this->options = array_merge($this->options, $options);
        }

        return new Client($this->options);
    }

    /**
     * Send the request.
     *
     * @param  string $method
     * @param  string $url
     * @param  array  $params
     * @return mixed|\Psr\Http\Message\ResponseInterface
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function request($method, $url, $params = [])
    {
        return $this->guzzle->request($method, $url, $params);
    }

    /**
     * The base64 encoded token for authentication.
     *
     * @return string
     */
    public function token()
    {
        return $this->auth->token();
    }
}
