<?php

namespace Ignite\Vendor\Helix;

use Ignite\Vendor\Helix\Models\Request\BillingCollection;
use Ignite\Vendor\Helix\Models\Request\BillingItem;
use Ignite\Vendor\Helix\Models\Request\OrderCollection;
use Ignite\Vendor\Helix\Models\Request\OrderItem;

final class Client implements ClientInterface
{
    /**
     * The underlying adapter to handle HTTP requests and responses.
     *
     * @var AdapterInterface
     */
    protected $adapter;

    /**
     * Client constructor.
     *
     * @param AdapterInterface $adapter
     */
    public function __construct(AdapterInterface $adapter)
    {
        $this->adapter = $adapter;
    }

    /**
     * The list of brands.
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getBrandList()
    {
        return $this->adapter->request('POST', 'get_brand_list');
    }

    /**
     * The list of categories.
     *
     * @param  string $countryCode
     * @param  string $languageCode
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getCategoryList($countryCode = 'US', $languageCode = 'eng')
    {
        $categoryId = 0;

        return $this->adapter->request('POST', 'get_category_list', [
            'form_params' => compact('categoryId', 'countryCode', 'languageCode')
        ]);
    }

    /**
     * The list of sub-categories.
     *
     * @param  int $categoryId
     * @param  string $countryCode
     * @param  string $languageCode
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getSubCategoryList($categoryId, $countryCode = 'US', $languageCode = 'eng')
    {
        return $this->adapter->request('POST', 'get_category_list', [
            'form_params' => compact('categoryId', 'countryCode', 'languageCode')
        ]);
    }

    /**
     * The list of products within a given category.
     *
     * @param  int $categoryId
     * @param  string $countryCode
     * @param  string $languageCode
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getProductList($categoryId, $countryCode = 'US', $languageCode = 'eng')
    {
        return $this->adapter->request('POST', 'category_products', [
            'form_params' => compact('categoryId', 'countryCode', 'languageCode')
        ]);
    }

    /**
     * The details of the given product.
     *
     * @param  int $productId
     * @param  string $countryCode
     * @param  string $languageCode
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getProductDetails($productId, $countryCode = 'US', $languageCode = 'eng')
    {
        return $this->adapter->request('POST', 'get_product_detail', [
            'form_params' => compact('productId', 'countryCode', 'languageCode')
        ]);
    }

    /**
     * The featured products for the site.
     *
     * @param  string $countryCode
     * @param  string $languageCode
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getFeaturedProducts($countryCode = 'US', $languageCode = 'eng')
    {
        return $this->adapter->request('POST', 'featured_product', [
            'form_params' => compact('countryCode', 'languageCode')
        ]);
    }

    /**
     * The special products for the site.
     *
     * @param  string $countryCode
     * @param  string $languageCode
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getSpecialProducts($countryCode = 'US', $languageCode = 'eng')
    {
        return $this->adapter->request('POST', 'special_product', [
            'form_params' => compact('countryCode', 'languageCode')
        ]);
    }

    /**
     * The new products for the site.
     *
     * @param  string $countryCode
     * @param  string $languageCode
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getNewProducts($countryCode = 'US', $languageCode = 'eng')
    {
        return $this->adapter->request('POST', 'new_products', [
            'form_params' => compact('countryCode', 'languageCode')
        ]);
    }

    /**
     * The new products for the site.
     *
     * @param  int $productId
     * @param  string $countryCode
     * @param  string $languageCode
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getSimilarProducts($productId, $countryCode = 'US', $languageCode = 'eng')
    {
        return $this->adapter->request('POST', 'similar_product', [
            'form_params' => compact('productId', 'countryCode', 'languageCode')
        ]);
    }

    /**
     * The details of the given order.
     *
     * @param  int $orderId
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function getOrderDetails($orderId)
    {
        //$orderId = new OrderId($orderId);
        return $this->adapter->request('POST', 'order-detail', [
            'form_params' => compact('orderId')
        ]);
    }

    /**
     * Place an order.
     *
     * @param array $billingItems
     * @param array $orderItems
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function placeOrder(array $billingItems, array $orderItems)
    {
        $billingItems = new BillingCollection(array_map(function ($item) {
            return new BillingItem($item);
        }, $billingItems));

        $orderItems = new OrderCollection(array_map(function ($item) {
            return new OrderItem($item);
        }, $orderItems));

        return $this->adapter->request('POST', 'third_party_order', [
            'form_params' => [
                'BILLING_DETAIL' => $billingItems->toArray(),
                'ORDER_ITEM' => $orderItems->toArray(),
            ]
        ]);
    }

    /**
     * Cancel the given order.
     *
     * @param  int $orderId
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function cancelOrder($orderId)
    {
        return $this->adapter->request('POST', 'cancel_order', [
            'form_params' => compact('orderId')
        ]);
    }
}