<?php

namespace Ignite\Vendor\Helix\Tests\Adapters;

use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Client;
use Ignite\Vendor\Helix\Adapters\Guzzle;
use Ignite\Vendor\Helix\Authentication;
use Ignite\Vendor\Helix\Tests\TestCase;
use Psr\Http\Message\ResponseInterface;

class GuzzleTest extends TestCase
{
    /**
     * @test
     */
    public function it_creates_a_correctly_configured_default_instance_of_a_client()
    {
        $auth = new Authentication($this->username, $this->password, $this->endpoint);
        $guzzle = new Guzzle($auth);

        $client = $guzzle->getDefaultInstance();

        $this->assertInstanceOf(Client::class, $client);
        $this->assertArrayHasKey('Content-Type', $client->getConfig('headers'));
        $this->assertEquals('application/x-www-form-urlencoded', $client->getConfig('headers')['Content-Type']);
        $this->assertEquals($auth->endpoint(), $client->getConfig('base_uri'));
        $this->assertEquals([$this->username, $this->password], $client->getConfig('auth'));
    }

    /**
     * @test
     */
    public function it_allows_client_options_to_be_overridden_on_the_default_client_via_constructor()
    {
        $auth = new Authentication($this->username, $this->password, $this->endpoint);
        $guzzle = new Guzzle($auth, null, ['auth' => ['baz', 'quux']]);

        $client = $guzzle->getDefaultInstance();

        $this->assertInstanceOf(Client::class, $client);
        $this->assertEquals(['baz', 'quux'], $client->getConfig('auth'));
    }

    /**
     * @test
     */
    public function it_allows_default_options_to_be_overridden_on_the_default_client_via_method()
    {
        $auth = new Authentication($this->username, $this->password, $this->endpoint);
        $guzzle = new Guzzle($auth, null, ['auth' => ['baz', 'quux']]);

        $client = $guzzle->getDefaultInstance(['auth' => ['garply', 'waldo']]);

        $this->assertInstanceOf(Client::class, $client);
        $this->assertEquals(['garply', 'waldo'], $client->getConfig('auth'));
    }

    /**
     * @test
     */
    public function it_allows_a_completely_custom_guzzle_client_implementation()
    {
        $container = [];
        $history = Middleware::history($container);
        $stack = HandlerStack::create();
        $stack->push($history);

        $auth = new Authentication($this->username, $this->password, $this->endpoint);
        $client = new Client([
            'base_uri' => 'http://httpbin.org',
            'handler' => $stack
        ]);
        $guzzle = new Guzzle($auth, $client);

        $response = $guzzle->request('GET', '/get');

        $this->assertInstanceOf(ResponseInterface::class, $response);
        $this->assertCount(1, $container);
        $this->assertArrayHasKey(0, $container);
        $this->assertArrayHasKey('request', $container[0]);
        $this->assertEquals('http://httpbin.org/get', $container[0]['request']->getUri()->__toString());
    }
}
