# Helix API

The repository provides a simple wrapper around the Helix Global Solutions API.

## Installation

Add the following to the repositories section of your `composer.json` file.

```
{
  "repositories" => {
    "helix" => {
      "type" => "composer",
      "url" => "https://satis.brightspot.work"
    }
  }
}
```

Then require the following package:

```
composer require ignite/helix-api
```

## Usage

The easiest and fastest way to instantiate the client object is to use the `Factory` class.

```php
<?php

use Ignite\Vendor\Helix\Factory;

$factory = new Factory();
// Pass in your account username, password, site_id and
// a boolean to indicate whether to use the live or sandbox API.
$client = $factory->make('username', 'password', 'site_id', $live = true);
```

Once you have a client instance, you are free to use any of the following methods:

#### getBrandsList()

```php
$brands = $client->getBrandsList();
// Returns:
[
  "status" => "success",
  "data" => [
    // ...
    [
      "brandId" => 1153,
      "brandName" => "1800 FLOWERS"
    ],
    // ...
  ],
  "message" => "Success"
]
```

#### getCategoryList($countryCode = 'US', $languageCode = 'eng')

```php
$categories = $client->getCategoryList();
// Returns:
[
    "status" => "success",
    "data" => [
        // ...
        [
            "categoryId" => 1,
            "categoryName_eng" => "Gift Cards",
            "siteId" => 20,
            "categoryParentId" => null
        ],
        // ...
    ],
    "message" => "success"
]
```

#### getSubCategoryList($categoryId, $countryCode = 'US', $languageCode = 'eng')

```php
$subcategories = $client->getSubCategoryList(2);
// Returns:
[
    "status" => "success",
    "data" => [
        // ...
        [
            "categoryId" => 36,
            "categoryName_eng" => "Audio",
            "siteId" => 20,
            "categoryParentId" => 2
        ],
        // ...
    ],
    "message" => "success"
]
```

#### getProductList($categoryId, $countryCode = 'US', $languageCode = 'eng')

```php
$products = $client->getProductList(1);
// Returns:
[
    "status" => "success",
    "data" => [
        // ...
        [
            "productId" => 683,
            "Active" => true,
            "productName_eng" => "$100 Amazon Gift Card",
            "productDescription_eng" => "<p style=\"margin: 0px; padding: 0px; font-family: Verdana, Arial, sans-serif; font-size: 11px; line-height: 16.5px; text-align: justify;\"><span style=\"line-height: 1.5em;\">Use your Amazon.ca Gift Card towards Books, Electronics, Music, and more. The Amazon.ca website is the place to find and discover almost anything you want to buy online at a great price!</span></p>\r\n<div><span style=\"line-height: 1.5em;\"><br /></span></div>",
            "productNumber" => "CE00401C",
            "inventory" => 4688,
            "productTypeId" => 13,
            "ecertificate" => 2,
            "productVideoUrl" => null,
            "brandName" => "Amazon",
            "isFeatured" => 0,
            "isSpecial" => 0,
            "supplierId" => 2,
            "brandId" => 32,
            "categoryName" => "Retail",
            "price" => 100,
            "PRODUIT_PHOTO_ID" => 661,
            "extensionImage" => "jpg",
            "productImageUrl" => "https://s3.ca-central-1.amazonaws.com/helix.staging.content/product_image/661.jpg",
            "prixArgent" => null,
            "point" => 0,
            "prixPointPromo" => null,
            "dateDebutPrixPointPromo" => null,
            "dateFinPrixPointPromo" => null,
            "prixPointOrderBy" => 0
        ],
        // ...
    ]
    "message" => "success"
]
```

#### getProductDetails($productId, $countryCode = 'US', $languageCode = 'eng')

```php
$product = $client->getProductDetails(683);
// Returns:
[
    "status" => "success",
    "data" => [
        [
            "productId" => 683,
            "Active" => true,
            "productName_eng" => "$100 Amazon Gift Card",
            "productDescription_eng" => "<p style=\"margin: 0px; padding: 0px; font-family: Verdana, Arial, sans-serif; font-size: 11px; line-height: 16.5px; text-align: justify;\"><span style=\"line-height: 1.5em;\">Use your Amazon.ca Gift Card towards Books, Electronics, Music, and more. The Amazon.ca website is the place to find and discover almost anything you want to buy online at a great price!</span></p>\r\n<div><span style=\"line-height: 1.5em;\"><br /></span></div>",
            "productNumber" => "CE00401C",
            "inventory" => 4688,
            "productTypeId" => 13,
            "ecertificate" => 2,
            "productVideoUrl" => null,
            "brandName" => "Amazon",
            "isFeatured" => 0,
            "isSpecial" => 0,
            "supplierId" => 2,
            "brandId" => 32,
            "categoryName" => "Retail",
            "price" => 100,
            "PRODUIT_PHOTO_ID" => 661,
            "extensionImage" => "jpg",
            "productImageUrl" => "https://s3.ca-central-1.amazonaws.com/helix.staging.content/product_image/661.jpg",
            "prixArgent" => null,
            "point" => 0,
            "prixPointPromo" => null,
            "dateDebutPrixPointPromo" => null,
            "dateFinPrixPointPromo" => null,
            "prixPointOrderBy" => 0
        ],
        [
            "images" => [
                [
                    "PRODUIT_PHOTO_ID" => 687,
                    "extensionImage" => "jpg"
                ]
            ]
        ]
    ],
    "message" => "success"
]
```

#### getFeaturedProducts($countryCode = 'US', $languageCode = 'eng')

```php
$products = $client->getFeaturedProducts();
// Returns:
[
    "status" => "success",
    "data" => [
        // ...
        [
            "productId" => 2217,
            "productNumber" => "KT06109U",
            "productModel" => "OCTGR3007",
            "productName_eng" => "Oster® Reversible Grill/Griddle",
            "productDescription_eng" => "<p>Features DuraCeramic™ non-stick coating that lasts 4 times longer. Grill cooks up to 20% faster saving energy and time. This is 2 appliances in 1 with double-sided reversible plate that is fully detachable. Has adjustable temperature control knob and cool-touch handles.</p>",
            "supplierId" => 2,
            "productImageId" => 1963,
            "productImageExtension" => "jpg",
            "price" => 99.49,
            "prixArgent" => null,
            "point" => 0,
            "prixPointPromo" => null,
            "dateDebutPrixPointPromo" => null,
            "dateFinPrixPointPromo" => null,
            "prixPointOrderBy" => 0,
            "productImageUrl" => "https://s3.ca-central-1.amazonaws.com/helix.staging.content/product_image/1963.jpg"
        ],
        // ...
    ],
    "message" => "success"
]
```

#### getSpecialProducts($countryCode = 'US', $languageCode = 'eng')

```php
$products = $client->getSpecialProducts();
// Returns:
[
    "status" => "success",
    "ParentCategory" => "Sale",
    "data" => [
        // ...
        [
            "productId" => 2217,
            "productNumber" => "KT06109U",
            "productModel" => "OCTGR3007",
            "productName_eng" => "Oster® Reversible Grill/Griddle",
            "productDescription_eng" => "<p>Features DuraCeramic™ non-stick coating that lasts 4 times longer. Grill cooks up to 20% faster saving energy and time. This is 2 appliances in 1 with double-sided reversible plate that is fully detachable. Has adjustable temperature control knob and cool-touch handles.</p>",
            "supplierId" => 2,
            "productImageId" => 1963,
            "productImageExtension" => "jpg",
            "price" => 99.49,
            "prixArgent" => null,
            "point" => 0,
            "prixPointPromo" => null,
            "dateDebutPrixPointPromo" => null,
            "dateFinPrixPointPromo" => null,
            "prixPointOrderBy" => 0,
            "productImageUrl" => "https://s3.ca-central-1.amazonaws.com/helix.staging.content/product_image/1963.jpg"
        ],
        // ...
    ],
    "message" => "success"
]
```

#### getNewProducts($countryCode = 'US', $languageCode = 'eng')

```php
$products = $client->getNewProducts();
// Returns:
[
    "status" => "success",
    "ParentCategory" => "New Products",
    "data" => [
        // ...
        [
            "productId" => 2217,
            "productNumber" => "KT06109U",
            "productModel" => "OCTGR3007",
            "productName_eng" => "Oster® Reversible Grill/Griddle",
            "productDescription_eng" => "<p>Features DuraCeramic™ non-stick coating that lasts 4 times longer. Grill cooks up to 20% faster saving energy and time. This is 2 appliances in 1 with double-sided reversible plate that is fully detachable. Has adjustable temperature control knob and cool-touch handles.</p>",
            "supplierId" => 2,
            "productImageId" => 1963,
            "productImageExtension" => "jpg",
            "price" => 99.49,
            "prixArgent" => null,
            "point" => 0,
            "prixPointPromo" => null,
            "dateDebutPrixPointPromo" => null,
            "dateFinPrixPointPromo" => null,
            "prixPointOrderBy" => 0,
            "productImageUrl" => "https://s3.ca-central-1.amazonaws.com/helix.staging.content/product_image/1963.jpg"
        ],
        // ...
    ],
    "message" => "success"
]
```

#### getSimilarProducts($productId, $countryCode = 'US', $languageCode = 'eng')

```php
$products = $client->getSimilarProducts(2217);
// Returns:
[
    "status" => "success",
    "data" => [
        // ...
        [
            "productId" => 5281,
            "productNumber" => "KT07118U",
            "productModel" => "60204",
            "productName_eng" => "Vitamix® - Professional 750 Heritage Blender Copper",
            "productDescription_eng" => "<p>5 Pre-programmed settings – Smoothies Hot soups Frozen desserts Purees and Self-cleaning. 2.2-Peak HP motor blends the toughest ingredients with the quietest motor. Easy cleaning with just a drop of soap. Laser-cut stainless steel hammermill and cutting blades. Cutting blades measure 4\" in diameter. Radial cooling fan and thermal protection system to keep from overheating. 6ft Power cord. 7-Year manufacturer full warranty. Includes: Low-profile 64oz container. Low-profile tamper. Getting Started Guide. Cookbook. DVD.</p>",
            "productNameFR" => null,
            "productName" => "Vitamix® - Professional 750 Heritage Blender Copper",
            "productDescription" => "<p>5 Pre-programmed settings – Smoothies Hot soups Frozen desserts Purees and Self-cleaning. 2.2-Peak HP motor blends the toughest ingredients with the quietest motor. Easy cleaning with just a drop of soap. Laser-cut stainless steel hammermill and cutting blades. Cutting blades measure 4\" in diameter. Radial cooling fan and thermal protection system to keep from overheating. 6ft Power cord. 7-Year manufacturer full warranty. Includes: Low-profile 64oz container. Low-profile tamper. Getting Started Guide. Cookbook. DVD.</p>",
            "supplierId" => 2,
            "productImageId" => 5135,
            "productImageExtension" => "jpg",
            "price" => 610.8,
            "prixArgent" => null,
            "point" => 0,
            "prixPointPromo" => null,
            "dateDebutPrixPointPromo" => null,
            "dateFinPrixPointPromo" => null,
            "prixPointOrderBy" => 0,
            "productImageUrl" => "https://s3.ca-central-1.amazonaws.com/helix.staging.content/product_image/5135.jpg"
        ],
        // ...
    ],
    "message" => "success"
]
```

#### getOrderDetails($orderId)

```php
$products = $client->getOrderDetails(194459);
// Returns:
[
    "status" => "success",
    "data" => [
        [
            "orderId" => 194459,
            "firstName" => "HElix",
            "lastName" => "test",
            "companyName" => "",
            "address" => "123 street",
            "address2" => "",
            "zip" => "12345",
            "city" => "new york",
            "state" => "NY",
            "stateId" => null,
            "countryId" => 2,
            "shipFirstName" => "HElix",
            "shipLastName" => "test",
            "shipCompanyName" => "",
            "shipAddress" => "123 street",
            "shipAddress2" => "",
            "shipZip" => "12345",
            "shipCity" => "new york",
            "shipState" => "NY",
            "shipStateId" => null,
            "shipCountryId" => 2,
            "isPaid" => true,
            "Active" => true,
            "createdDate" => "2019-03-12T21:34:14.823Z",
            "comments" => null,
            "phone" => "",
            "billEmail" => "",
            "subTotal" => 1922.82,
            "Total" => 1922.82,
            "billCountryName" => "United States",
            "billCountryCode" => "US",
            "shipCountryName" => "United States",
            "shipCountryCode" => "US",
            "billStateName" => "NY",
            "billStateISOCode" => "NY",
            "shipStateName" => "NY",
            "shipStateISOCode" => "NY",
            "lineNumber" => 215504,
            "productId" => 2995,
            "productName" => "Garmin™ Approach S60 - Black Band",
            "qty" => 2,
            "productNumber" => "EL11067U",
            "price" => 371.82,
            "orderStatus" => "Cancelled",
            "waybill" => null,
            "confNum" => null,
            "shipdate" => null
        ],
        [
            "orderId" => 194459,
            "firstName" => "HElix",
            "lastName" => "test",
            "companyName" => "",
            "address" => "123 street",
            "address2" => "",
            "zip" => "12345",
            "city" => "new york",
            "state" => "NY",
            "stateId" => null,
            "countryId" => 2,
            "shipFirstName" => "HElix",
            "shipLastName" => "test",
            "shipCompanyName" => "",
            "shipAddress" => "123 street",
            "shipAddress2" => "",
            "shipZip" => "12345",
            "shipCity" => "new york",
            "shipState" => "NY",
            "shipStateId" => null,
            "shipCountryId" => 2,
            "isPaid" => true,
            "Active" => true,
            "createdDate" => "2019-03-12T21:34:14.823Z",
            "comments" => null,
            "phone" => "",
            "billEmail" => "",
            "subTotal" => 1922.82,
            "Total" => 1922.82,
            "billCountryName" => "United States",
            "billCountryCode" => "US",
            "shipCountryName" => "United States",
            "shipCountryCode" => "US",
            "billStateName" => "NY",
            "billStateISOCode" => "NY",
            "shipStateName" => "NY",
            "shipStateISOCode" => "NY",
            "lineNumber" => 215505,
            "productId" => 4267,
            "productName" => "Sony 7.1.2ch 800W Dolby Atmos® Sound Bar",
            "qty" => 1,
            "productNumber" => "EL11056U",
            "price" => 1179.18,
            "orderStatus" => "Cancelled",
            "waybill" => null,
            "confNum" => null,
            "shipdate" => null
        ]
    ],
    "message" => "success"
]
```

#### placeOrder(array $billingItems, array $orderItems)

```php
$products = $client->placeOrder(
    [
        [
            'first_name' => 'John',
            'last_name' => 'Smith',
            'email_id' => 'foo',
            'company_name' => 'Brightspot',
            'address1' => '102 Decker Ct',
            'address2' => 'Suite 150',
            'postal_code' => '75062',
            'city' => 'Irving',
            'state' => 'TX',
            'countryCode' => 'US',
            'ip_address' => '47.190.18.3'
        ]
    ],
    [
        ['product_id' => 2995, 'qty' => 2]
    ]
);
// Returns:
[
    "status" => "success",
    "COMMANDE_ID" => 194586,
    "data" => [],
    "message" => "Order placed successfully"
]
```

#### cancelOrder($orderId)

```php
$products = $client->cancelOrder(194459)
// Result
[
    "status" => "success",
    "message" => "Success"
]
```

## Developers Documentation

#### Table of Contents
- [Factory](#factory)
- [Authentication](#authentication)
- [Client](#client)
- [Adapters](#adapters)
- [Decorators](#decorators)
- [Transformers](#transformers)

---

### Factory

The responsibility of a `Factory` class is to build other instances of classes. In the Helix client, we use a `Factory` class to build an instance of a `Client` which is "decorated" with a JSON `Transformer`, more on those terms later. For now, it's best to understand that the `Factory` is the primary mechanism of creating a `Client` instance because it understands which parts fit together and how they should be constructed.

#### Making a Client with the default Adapter and Decorator

In order to construct a `Factory` object, you can simply 'new up' an instance as follows:

```php
use \Ignite\Vendor\Helix\Factory;

$factory = new Factory();
$client = $factory->make($username, $password, $siteId, $liveMode);
```

#### Making a Client with a custom Adapter and/or Decorator

If you have custom `Adapter` or `Decorator` classes that you would like to register, you can pass them as follows:

```php
$adapters = [
    'my_adapter' => \Fully\Qualified\Class\Name::class
];
$decorators = [
    'my_decorator' => \Fully\Qualified\Class\Name::class
];
$factory = new Factory($adapters, $decorators);
```

Now that you've registered your custom `Adapter` and `Decorator` classes, you can reference them when making a new `Client` instance as follows:

```php
$client = $factory->make($username, $password, $siteId, $liveMode, 'my_adapter', 'my_decorator');
$client = $factory->make($username, $password, $siteId, $liveMode, '', 'my_decorator');
$client = $factory->make($username, $password, $siteId, $liveMode, 'my_adapter', '');
```

### Authentication

Before we can begin using the `Client`, we must authenticate our requests. In order to do so, we must tell our `Adapter` which `URI` to use and which username and password to use for credentials.

```php
$auth = new Authentication($username = 'my_username', $password = 'my_password', $siteId = 20, $liveMode = true);
```

The `Authentication` class can provide some very useful specifics for our `Adapter` classes. For example, we can determine whether we are in debug mode by calling `$auth->debug()`. We can also generate the Basic Auth header's `Bearer` token using `$auth->token()`. We can access `$auth->username()` and `$auth->password()`. Most importantly, we can access the API endpoint using `$auth->uri()`.

Each `Adapter` class should accept the `Authentication` class via their constructor. Once the data in the `Authentication` class has been assigned to the `Adapter`, we don't need the class again.

### Client

The `Client` class is responsible for proxying the 'high-level' requests to the Helix API via an `Adapter`, which understands the 'low-level' actions required to send the correct data over the wire to the Helix application.  The `Client` implements the following methods as outlined by the `\Ignite\Vendor\Helix\ClientInterface`.

```php
interface ClientInterface
{
    public function getBrandList();
    public function getCategoryList($countryCode = 'US', $languageCode = 'eng');
    public function getSubCategoryList($categoryId, $countryCode = 'US', $languageCode = 'eng');
    public function getProductList($categoryId, $countryCode = 'US', $languageCode = 'eng');
    public function getProductDetails($productId, $countryCode = 'US', $languageCode = 'eng');
    public function getFeaturedProducts($countryCode = 'US', $languageCode = 'eng');
    public function getSpecialProducts($countryCode = 'US', $languageCode = 'eng');
    public function getNewProducts($countryCode = 'US', $languageCode = 'eng');
    public function getSimilarProducts($productId, $countryCode = 'US', $languageCode = 'eng');
    public function getOrderDetails($orderId);
    public function placeOrder(array $billingItems, array $orderItems);
    public function cancelOrder($orderId);
}
```

In order to create a `Client` instance, you are required to pass in a concrete instance of an `\Ignite\Vendor\Helix\AdapterInterface`. By default, we use the `Guzzle` adapter.

```php
use Ignite\Vendor\Helix\Adapters\Guzzle;
use Ignite\Vendor\Helix\Client;

$adapter = new Guzzle($authentication);
$client = new Client($adapter)
```

When you make a request using any of the methods above, you will receive an instance of the PSR-7 Response Interface. (e.g. `\Psr\Http\Message\ResponseInterface`)

### Adapters

The `Adapter` class is responsible for sending the data over the wire to the Helix API.

#### Using a custom Adapter
By default we use the Guzzle PHP library to send cURL requests. However, you can swap out `Guzzle` for your own custom adapter by implementing `\Ignite\Vendor\Helix\AdapterInterface`. The `AdapterInterface` looks as follows:

```php
interface AdapterInterface
{
    public function request($method, $url, $params = []);
}
```

The `request()` method must return an instance of `\Psr\Http\Message\ResponseInterface`. The PHP community has standardised Http requests/responses to a point where it is no longer acceptable/pragmatic not to implement PSR-7.

### Decorators

The `Client` class is not responsible for transforming the source JSON string representing the response data. In order to transform the response, we need to use a `Decorator`.

By default, we use `\Ignite\Vendor\Helix\Decorators\Json`. Since the Helix API will only return JSON, if we want to parse the JSON, we will always need to 'decorate' our response using this class.

```php
use Ignite\Vendor\Helix\Adapters\Guzzle;
use Ignite\Vendor\Helix\Client;
use Ignite\Vendor\Helix\Decorators\Json;

$adapter = new Guzzle($authentication);
$client = new Client($adapter)
$decorator = new Json($client);
```

The `\Ignite\Vendor\Helix\Decorators\Json` decorator inherits from `\Ignite\Vendor\Helix\AbstractDecorator` which implements a 'magic' `__call` method. The `__call` method proxies calls on the concrete `Decorator` class

The great thing about decorators is that we can stack different implementations on top of each other. For example, if for some reason we needed to convert every response directly into XML, we could create a decorator that first uses the JSON transformer and then uses an XML transformer. An example should illustrate the concept a little better.

```php
use Psr\Http\Message\ResponseInterface;

class XmlTransformer
{
    public function transform(ResponseInterface $response)
    {
        $array = (new JsonTransformer)->transform($response);

        $xml = new SimpleXMLElement('<response/>');
        array_walk_recursive($array, [$xml, 'addChild']);
        return $xml->asXML();
    }
}

class Xml extends AbstractDecorator
{
    public function decorate(ResponseInterface $response)
    {
        return (new XmlTransformer)->transform($response);
    }
}
```

### Transformers

The `Transformer` classes are responsible for transforming the response body into something usable. They act as a service layer between the `Decorator` and the `Client` to decouple the responsibility of parsing data from one form into another. They should implement the `\Ignite\Vendor\Helix\TransformerInterface`.

```php
interface TransformerInterface
{
    public function transform(ResponseInterface $response);
}
```

### Using in the Laravel Framework

When you use a dependency injection container, it can be useful to create only one instance of the `Authentication` class and load the necessary credentials in from configuration. For example, in Laravel, you might do something as follows:

```php
// File: config/helix.php
return [
    'username' => env('HELIX_API_USERNAME', ''),
    'password' => env('HELIX_API_PASSWORD', ''),
    'site_id' => env('HELIX_API_SITE_ID', ''),
    'use_live' => env('HELIX_API_USE_LIVE', ''),
    'catalog_id' => env('HELIX_API_CATALOG_ID', ''),
    'access_token' => env('HELIX_API_ACCESS_TOKEN', ''),
    'adapters' => [
        // Add custom adapters here
    ],
    'decorators' => [
        // Add custom decorators here
    ],
];

```php
// File: app/Providers/HelixServiceProvider.php

namespace App\Providers;

use Ignite\Vendor\Helix\Factory;
use Illuminate\Support\ServiceProvider;

class AppServiceProvider extends ServiceProvider
{
    public function register()
    {
        $this->app->singleton('helix', function (Application $app) {
            $config = $app['config']->get('helix');

            $factory = new Factory($config['adapters'], $config['decorators'])
            return $factory->make(
                $config['username'],
                $config['password'],
                $config['site_id'],
                $config['use_live']
            );
        });
    }
}
```

Then you can make requests using:

```php
app('helix')->getBrandsList();
```

## Todos

```
- [x] Write tests
- [x] Write documentation
- [ ] Consult with Helix Technical Team
- [ ] Push to Bitbucket
- [ ] Push to Brightspot Satis
- [ ] Tag version 1.0
- [ ] Write a Laravel Bridge package
```
