<?php

namespace Ignite\Vendor\Helix\Laravel\Tests\Unit\Response;

use Ignite\Vendor\Helix\Laravel\Response;
use Ignite\Vendor\Helix\Tests\TestCase;
use Illuminate\Support\Collection;

class ResponseTest extends TestCase
{
    /**
     * @test
     */
    public function it_can_determine_the_status_of_the_response()
    {
        $response = new Response([
            'status' => 'error'
        ]);

        $this->assertEquals('error', $response->status());
    }

    /**
     * @test
     */
    public function it_can_determine_whether_the_request_passed()
    {
        $response = new Response([
            'status' => 'success'
        ]);

        $this->assertTrue($response->passed());
    }

    /**
     * @test
     */
    public function it_can_determine_whether_the_request_failed()
    {
        $response = new Response([
            'status' => 'error'
        ]);

        $this->assertTrue($response->failed());
    }

    /**
     * @test
     */
    public function it_can_get_the_response_message()
    {
        $response = new Response([
            'message' => 'Success'
        ]);

        $this->assertEquals('Success', $response->message());
    }

    /**
     * @test
     */
    public function it_can_get_an_arbitrary_response_value_by_key()
    {
        $response = new Response([
            'COMMANDE_ID' => 123131
        ]);

        $this->assertEquals(123131, $response->get('COMMANDE_ID'));
    }

    /**
     * @test
     */
    public function it_can_get_a_default_value_when_an_arbitrary_response_value_by_key_is_missing()
    {
        $response = new Response([]);

        $this->assertEquals('foo', $response->get('COMMANDE_ID', 'foo'));
    }

    /**
     * @test
     */
    public function it_can_determine_if_the_response_has_a_given_value()
    {
        $response = new Response([]);

        $this->assertFalse($response->hasValue('foo'));

        $response = new Response(['foo' => 'bar']);

        $this->assertTrue($response->hasValue('foo'));
    }

    /**
     * @test
     */
    public function it_can_determine_if_the_response_doesnt_have_a_given_value()
    {
        $response = new Response([]);

        $this->assertTrue($response->doesntHaveValue('foo'));

        $response = new Response(['foo' => 'bar']);

        $this->assertFalse($response->doesntHaveValue('foo'));
    }

    /**
     * @test
     */
    public function it_can_transform_response_data_to_collection()
    {
        $response = new Response([
            'data' => [
                [
                    'productId' => 5340,
                    'productNumber' => 'EL11514U',
                    'productModel' => '316238',
                ]
            ]
        ]);

        $this->assertInstanceOf(Collection::class, $response->data());
    }

    /**
     * @test
     */
    public function it_will_proxy_methods_to_the_underlying_data_collection()
    {
        $response = new Response([
            'data' => [
                [
                    'productId' => 5340,
                    'productNumber' => 'EL11514U',
                    'productModel' => '316238',
                ]
            ]
        ]);

        $proxy = $response->pluck('productModel');
        $this->assertInstanceOf(Collection::class, $proxy);
        $this->assertEquals('316238', $proxy->first());
    }

    /**
     * @test
     */
    public function it_will_throw_an_exception_when_cannot_proxy_methods_to_the_underlying_data_collection()
    {
        $this->expectException(\Exception::class);

        $response = new Response([
            'data' => [
                [
                    'productId' => 5340,
                    'productNumber' => 'EL11514U',
                    'productModel' => '316238',
                ]
            ]
        ]);

        $proxy = $response->foo();
    }

    /**
     * @test
     */
    public function it_can_convert_the_response_to_an_array()
    {
        $response = new Response(['foo' => 'bar']);

        $result = $response->toArray();

        $this->assertTrue(is_array($result));
        $this->assertArrayHasKey('foo', $result);
        $this->assertEquals('bar', $result['foo']);
    }

    /**
     * @test
     */
    public function it_can_convert_the_response_to_a_json_string()
    {
        $response = new Response(['foo' => 'bar']);

        $result = $response->toJson();

        $this->assertTrue(is_string($result));
        $this->assertEquals('{"foo":"bar"}', $result);
    }

    /**
     * @test
     */
    public function it_can_cast_the_response_to_a_string_which_will_render_as_json()
    {
        $response = new Response(['foo' => 'bar']);

        $result = (string) $response;

        $this->assertTrue(is_string($result));
        $this->assertEquals('{"foo":"bar"}', $result);
    }
}
