<?php

namespace Ignite\Vendor\Helix\Laravel\Tests\Unit\Sso;

use Ignite\Vendor\Helix\Laravel\Contracts\DecoratableSsoClientInterface;
use Ignite\Vendor\Helix\Laravel\Response;
use Ignite\Vendor\Helix\Laravel\Sso\Client;
use Ignite\Vendor\Helix\Laravel\Sso\Payload;
use Ignite\Vendor\Helix\Laravel\Sso\Session;
use Ignite\Vendor\Helix\Laravel\Tests\Stubs\User;
use Ignite\Vendor\Helix\Laravel\Tests\TestCase;
use Mockery;

class SessionTest extends TestCase
{
    /**
     * @test
     */
    public function it_will_request_a_sso_login_url_for_the_given_user_with_an_existing_helix_user_id()
    {
        $sessionKey = config('helix.sso.session.key');
        $helixIdAttribute = config('helix.user.attributes.helix_user_id');

        $response = $this->getResponseData([
            'helixUserId' => $helixUserId = '690901',
            'URL' => $loginUrl = "http://52.60.111.154/brightspot/login/D5F9350EF38FC919E684023E68670584F951E3216391E53F28CDB64274D1C1EC"
        ]);

        $payload = new Payload($this->getPayloadData([
            'helixUserId' => $helixUserId
        ]));

        request()->setLaravelSession(session());
        $request = request();

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signIn')->once()->andReturn(new Response($response));

        $user = $this->getMockUser($payload);
        //$user->shouldReceive('jsonSerialize')->once()->andReturn($payload); // Only use when testing error logging!
        $user->shouldReceive('getAttribute')->once()->with($helixIdAttribute)->andReturn($payload['helixUserId']);
        $user->shouldReceive('update')->once();

        $session = new Session($client);
        $session->handle($request, $user);

        $this->assertEquals($loginUrl, $request->session()->get($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_request_a_sso_login_url_for_the_given_user_without_an_existing_helix_user_id()
    {
        $sessionKey = config('helix.sso.session.key');
        $helixIdAttribute = config('helix.user.attributes.helix_user_id');

        $response = $this->getResponseData([
            'helixUserId' => $helixUserId = '690901',
            'URL' => $loginUrl = "http://52.60.111.154/brightspot/login/D5F9350EF38FC919E684023E68670584F951E3216391E53F28CDB64274D1C1EC"
        ]);
        $payload = new Payload($this->getPayloadData()); // No helixUserId attribute

        request()->setLaravelSession(session());
        $request = request();

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signUp')->once()->andReturn(new Response($response));

        $user = $this->getMockUser($payload);
        //$user->shouldReceive('jsonSerialize')->once()->andReturn($payload);
        $user->shouldReceive('getAttribute')->once()->with($helixIdAttribute)->andReturn($payload['helixUserId']);
        $user->shouldReceive('update')->once();

        $session = new Session($client);
        $session->handle($request, $user);

        $this->assertEquals($loginUrl, $request->session()->get($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_add_an_error_code_to_the_session_when_the_payload_validation_fails()
    {
        $sessionKey = config('helix.sso.session.key');
        $helixIdAttribute = config('helix.user.attributes.helix_user_id');

        $user = new User([
            $helixIdAttribute => $helixUserId = '690901',
            'email' => 'bad email address'
        ]);

        request()->setLaravelSession(session());

        $client = app(DecoratableSsoClientInterface::class);

        $session = new Session($client);
        $session->handle(request(), $user);

        $this->assertEquals(Client::SSO_INVALID_DATA, session($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_add_an_error_code_to_the_session_when_the_sso_response_contains_an_error()
    {
        $sessionKey = config('helix.sso.session.key');
        $helixIdAttribute = config('helix.user.attributes.helix_user_id');

        $response = $this->getResponseData([
            "code" => 500,
            "status" => "error",
            "message" => "Missing parameters: foo"
        ]);
        $payload = new Payload($this->getPayloadData([
            'helixUserId' => $helixUserId = '690901'
        ]));

        request()->setLaravelSession(session());

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signIn')->once()->andReturn(new Response($response));

        $user = $this->getMockUser($payload);
        $user->shouldReceive('jsonSerialize')->once()->andReturn($payload->toJson()); // So the logger can serialize it.
        $user->shouldReceive('getAttribute')->once()->with($helixIdAttribute)->andReturn($payload['helixUserId']);

        $session = new Session($client);
        $session->handle(request(), $user);

        $this->assertEquals(Client::SSO_INVALID_RESPONSE, session($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_add_an_error_code_to_the_session_when_the_sso_login_url_is_missing()
    {
        $sessionKey = config('helix.sso.session.key');
        $helixIdAttribute = config('helix.user.attributes.helix_user_id');

        $response = $this->getResponseData([
            'URL' => $loginUrl = ""
        ]);
        $payload = new Payload($this->getPayloadData([
            'helixUserId' => $helixUserId = '690901'
        ]));

        request()->setLaravelSession(session());

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signIn')->once()->andReturn(new Response($response));

        $user = $this->getMockUser($payload);
        $user->shouldReceive('jsonSerialize')->once()->andReturn($payload->toJson()); // So the logger can serialize it.
        $user->shouldReceive('getAttribute')->once()->with($helixIdAttribute)->andReturn($payload['helixUserId']);

        $session = new Session($client);
        $session->handle(request(), $user);

        $this->assertEquals(Client::SSO_INVALID_LOGIN_URL, session($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_add_an_error_code_to_the_session_when_the_sso_login_url_is_malformed()
    {
        $sessionKey = config('helix.sso.session.key');
        $helixIdAttribute = config('helix.user.attributes.helix_user_id');

        $response = $this->getResponseData([
            'helixUserId' => $helixUserId = '690901',
            'URL' => $loginUrl = "badly formed url"
        ]);
        $payload = new Payload($this->getPayloadData([
            'helixUserId' => $helixUserId
        ]));

        request()->setLaravelSession(session());

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signIn')->once()->andThrow(new Response($response));

        $user = $this->getMockUser($payload);
        $user->shouldReceive('jsonSerialize')->once()->andReturn($payload->toJson()); // So the logger can serialize it.
        $user->shouldReceive('getAttribute')->once()->with($helixIdAttribute)->andReturn($payload['helixUserId']);
        //$user->shouldReceive('update')->once();

        $session = new Session($client);
        $session->handle(request(), $user);

        $this->assertEquals(Client::SSO_INVALID_LOGIN_URL, session($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_add_an_error_code_to_the_session_when_a_general_error_occurs()
    {
        $sessionKey = config('helix.sso.session.key');

        $payload = new Payload($this->getPayloadData([
            'helixUserId' => $helixUserId = '690901'
        ]));

        request()->setLaravelSession(session());

        $client = Mockery::mock(DecoratableSsoClientInterface::class);

        $user = $this->getMockUser($payload);
        $user->shouldReceive('jsonSerialize')->once()->andReturn($payload->toJson());

        $session = new Session($client);
        $session->handle(request(), $user);

        $this->assertEquals(Client::SSO_UNKNOWN_ERROR, session($sessionKey));
    }
}
