<?php

namespace Ignite\Vendor\Helix\Laravel\Services;

use Ignite\Vendor\Helix\Laravel\Exceptions\OpenSslCipherNotSupported;

class Token
{
    /**
     * @var string
     */
    protected $key;

    /**
     * @var string
     */
    protected $decodedKey;

    /**
     * @var string
     */
    protected $cipher;

    /**
     * Token constructor.
     *
     * @param $key
     * @param string $cipher
     *
     * @throws \Exception
     */
    public function __construct($key, $cipher = 'AES-128-ECB')
    {
        $this->guardAgainstMissingCipherMethod($cipher);
        $this->key = $key;
        $this->decodedKey = base64_decode($key);
        $this->cipher = $cipher;
    }

    /**
     * Guard against configured openSSL cipher method missing on system.
     *
     * @param $cipher
     *
     * @throws \Exception
     */
    protected function guardAgainstMissingCipherMethod($cipher)
    {
        if (! in_array(strtolower($cipher), openssl_get_cipher_methods())) {
            throw new OpenSslCipherNotSupported($cipher);
        }
    }

    /**
     * The initialization vector, also called a nonce, is an arbitrary number that can be used along with a secret key.
     *
     * @return string
     */
    protected function getInitializationVector()
    {
        // AES-128-ECB may return 0, so it should be deprecated; Helix will need to update
        $length = openssl_cipher_iv_length($this->cipher);

        // version 7.4 and above, openssl_random_pseudo_bytes() will throw a warning if length is 0
        // whereas 7.3 and below just returns empty string, which we want to emulate
        if (0 === $length
            && (PHP_MAJOR_VERSION > 7 || (PHP_MAJOR_VERSION == 7 && PHP_MINOR_VERSION >= 4))
        ) {
            return '';
        }

        return (string) openssl_random_pseudo_bytes($length);
    }

    /**
     * Encrypt the user id to create the access token.
     *
     * @param string $catalogToken
     * @param string $userId
     *
     * @return string
     */
    public function encrypt($catalogToken, $userId)
    {
        $iv = $this->getInitializationVector();

        $token = openssl_encrypt("{$catalogToken}-{$userId}", $this->cipher, $this->decodedKey, OPENSSL_RAW_DATA, $iv);

        return strtoupper(html_entity_decode(bin2hex($token), ENT_COMPAT | ENT_QUOTES, 'UTF-8'));
    }

    /**
     * Decrypt the access token and find the user id.
     *
     * @param string $token
     *
     * @return int
     */
    public function decrypt($token)
    {
        $iv = $this->getInitializationVector();

        return openssl_decrypt(hex2bin($token), $this->cipher, $this->decodedKey, OPENSSL_RAW_DATA, $iv);
    }
}
