<?php

return [
    'enabled' => env('HELIX_BRIDGE_ENABLED', true),

    /**
     * The user information configured here defines the model and attributes to be used by
     * the HTTP routes and event listeners in this package in order to determine the
     * user model to use to query for helix specific data in an abstract way.
     */
    'user' => [
        /**
         * The user model to query when verifying points and authorizing a user with
         * Helix. The model value should be the fully qualified class name of an
         * eloquent class that implements the ProvidesHelixPayload interface
         * and the Payloadable trait to transform user data for Helix.
         *
         * @see \Ignite\Vendor\Helix\Laravel\Contracts\ProvidesHelixPayload
         * @see \Ignite\Vendor\Helix\Laravel\Resources\Concerns\Payloadable
         * @see \Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\RetrievesUserModel
         */
        'model' => \App\User::class,

        /**
         * The following attributes help the package to identify the data attributes
         * required to match user id values and determine the number of points
         * for a user/participant and verify the provided access token.
         *
         * @see \Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\RetrievesUserIdAttribute
         * @see \Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\RetrievesPointsAttribute
         * @see \Ignite\Vendor\Helix\Laravel\Listeners\Concerns\RetrievesHelixUserIdAttribute
         */
        'attributes' => [
            'id' => 'id',
            'points' => 'points',
            'helix_user_id' => 'helix_user_id'
        ]
    ],

    /**
     * The following configuration defines the shared token values for use in a Helix
     * access token to be encrypted and decrypted securely. The Token service will
     * verify that the cipher is available on the server or throw an exception.
     *
     * @see \Ignite\Vendor\Helix\Laravel\Services\Token
     * @see \Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\GuardAgainstInvalidToken
     */
    'token' => [
        /**
         * A 16 character alpha-numeric randomg string. The shared catalog token is
         * typically defined by Helix during the client onboarding phase. When
         * setting up a new integration, we should contact Hassan Tariq.
         */
        'catalog' => env('HELIX_TOKEN_CATALOG'),

        /**
         * A cryptographically secure, base64 encoded shared key will be provided by
         * Helix and will be used to encrypt and decrypt the access token sent in
         * requests going to and coming from Helix. Keep it secret. Keep it safe.
         */
        'key' => env('HELIX_TOKEN_KEY', ''),

        /**
         * The encryption algorithm used is AES-128-ECB, this specific algorithm was
         * mandated by Helix. If the cipher type ever changes, we would also need
         * to generate compatible shared key of the correct number of bytes.
         */
        'cipher' => env('HELIX_TOKEN_CIPHER', 'AES-128-ECB'),
    ],

    /**
     * The Helix catalog ID is a UUID generated by Brightspot and sent to Helix in
     * order to identify the catalog between our two systems. The value will be
     * sent with every SSO request and will be included in many responses too.
     */
    'catalog' => env('HELIX_CATALOG_ID', ''),

    /**
     * The memberGroupId is a unique id per program for cases when Helix wants
     * to use one site id for multiple catalogs or programs.
     */
    'memberGroupId' => env('HELIX_MEMBERGROUP_ID', null),

    /**
     * The single sign on configuration defines the API endpoint, username and password
     * used when we attempt to sign into Helix from our application. We also define
     * the components to handle the request and response as well as session data
     * and event listeners which handle resolving the user payload during the
     * sign-in and sign-up processes.
     *
     * @see \Ignite\Vendor\Helix\Laravel\Sso\Client
     * @see \Ignite\Vendor\Helix\Laravel\Sso\Payload
     * @see \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnLogin
     * @see \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnRegister
     * @see \Ignite\Vendor\Helix\Laravel\Contracts\SsoClientInterface
     * @see \Ignite\Vendor\Helix\Laravel\Contracts\DecoratableSsoClientInterface
     * @see \Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstMissingLoginUrl
     * @see \Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstFailedSsoRequest
     * @see \Ignite\Vendor\Helix\Laravel\Exceptions\SsoRequestFailed
     * @see \Ignite\Vendor\Helix\Laravel\Exceptions\SsoBadLoginUrl
     */
    'sso' => [
        /**
         * The Helix single sign on endpoint URL is provided by Helix and should be
         * located at our custom subdomain for Brightspot. In order to find out
         * exactly what the value should be, we should contact Hassan Tariq.
         */
        'endpoint' => env('HELIX_SSO_ENDPOINT', ''),

        /**
         * The Helix single sign on username is provided by Helix. In order to find
         * out exactly what the value should be, we should contact Hassan Tariq.
         */
        'username' => env('HELIX_SSO_USERNAME', ''),

        /**
         * The Helix single sign on password is provided by Helix. In order to find
         * out exactly what the value should be, we should contact Hassan Tariq.
         */
        'password' => env('HELIX_SSO_PASSWORD', ''),

        /**
         * The Laravel session information required to store the single sign on login
         * URL consistently to the correct key. The event listeners will store the
         * value and the view composer will make it available for display.
         *
         * @see \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnLogin
         * @see \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnRegister
         * @see \Ignite\Vendor\Helix\Laravel\Http\Views\Composers\IframeComposer
         */
        'session' => [
            /**
             * The string to use as the key when storing single sign on information
             * in the session. Defining it here makes it easy for our application
             * to consume the key consistently through out the SSO lifecycle.
             */
            'key' => env('HELIX_SSO_SESSION_KEY', 'helix_login_url'),
        ],

        /**
         * The single sign on event listeners to be registered with the correct events
         * via the Helix service provider. The single sign on synchronization code
         * is triggered by the user login event and user registration event.
         *
         * @see \Ignite\Vendor\Helix\Laravel\HelixServiceProvider
         * @see \Illuminate\Auth\Events\Login
         * @see \Illuminate\Auth\Events\Registered
         */
        'listeners' => [
            /**
             * The user registered event is a Laravel core event, if you're not using
             * the core auth scaffolding, please make sure that you are dispatching
             * the registered event from your custom auth controller code.
             */
            'registered' => [
                \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnRegister::class,
            ],
            /**
             * The user login event is a Laravel core event, if you're not using the
             * core auth scaffolding, please make sure that you are dispatching
             * the login event from your custom auth controller code.
             */
            'login' => [
                \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnLogin::class,
            ]
        ],

        /**
         * The Helix adapter to use. By default we use the Guzzle Adapter since it already
         * implements the PSR-7 Http Standard Interfaces. If Guzzle was ever deprecated
         * we could swap the implementation out very easily from within this project.
         */
        'adapter' => 'guzzle',

        /**
         * The Helix decorator to use. By default we use the JSON decorator, JSON is the
         * easiest data format to use with the Helix API. However, we could swap out
         * the implementation easily if we ever needed to transform the response.
         */
        'decorator' => 'json',

        /**
         * The default adapters can be changed by merging the array below. The list
         * of adapter definitions must be in a consistent key-value pair format,
         * for example: 'short_name' => \Fully\Qualified\Class\Name::class
         */
        'adapters' => [
            'guzzle' => \Ignite\Vendor\Helix\Adapters\Guzzle::class
        ],

        /**
         * The default decorators can be changed by merging the array below. The list
         * of decorators definitions must be in a consistent key-value pair format,
         * for example: 'short_name' => \Fully\Qualified\Class\Name::class
         */
        'decorators' => [
            'json' => \Ignite\Vendor\Helix\Laravel\Decorators\Laravel::class
        ]
    ],

    'api' => [
        /**
         * The Helix API endpoint URL is provided by Helix and should be located at
         * our custom subdomain for Brightspot but may be different to the SSO
         * endpoint. We should contact Hassan Tariq to ascertain this value.
         */
        'endpoint' => env('HELIX_API_ENDPOINT', ''),

        /**
         * The Helix API username is provided by Helix but may be different to the
         * SSO username. In order to find out exactly what the value should be,
         * we should contact Hassan Tariq.
         */
        'username' => env('HELIX_API_USERNAME', ''),

        /**
         * The Helix API password is provided by Helix but may be different to the
         * SSO password. In order to find out exactly what the value should be,
         * we should contact Hassan Tariq.
         */
        'password' => env('HELIX_API_PASSWORD', ''),

        /**
         * The Helix adapter to use. By default we use the Guzzle Adapter since it already
         * implements the PSR-7 Http Standard Interfaces. If Guzzle was ever deprecated
         * we could swap the implementation out very easily from within this project.
         */
        'adapter' => 'guzzle',

        /**
         * The Helix decorator to use. By default we use the JSON decorator, JSON is the
         * easiest data format to use with the Helix API. However, we could swap out
         * the implementation easily if we ever needed to transform the response.
         */
        'decorator' => 'json',

        /**
         * The default adapters can be changed by merging the array below. The list
         * of adapter definitions must be in a consistent key-value pair format,
         * for example: 'short_name' => \Fully\Qualified\Class\Name::class
         */
        'adapters' => [
            'guzzle' => \Ignite\Vendor\Helix\Adapters\Guzzle::class
        ],

        /**
         * The default decorators can be changed by merging the array below. The list
         * of decorators definitions must be in a consistent key-value pair format,
         * for example: 'short_name' => \Fully\Qualified\Class\Name::class
         */
        'decorators' => [
            'json' => \Ignite\Vendor\Helix\Laravel\Decorators\Laravel::class
        ]
    ],

    /**
     * The list of carriers and association data can be looked up by key using the carrier
     * code and iso2 country code, separated by a dash. When looking up a carrier, it's
     * best to use the Factory class which includes a helper method to format the key.
     */
    'carriers' => [
        'CLAIT' => [
            'code'  => 'CLAIT',
            'name'  => 'Chronopost International',
            'url'   => 'https://www.chronopost.fr/tracking-no-cms/suivi-page',
            'query' => '?listeNumerosLT='
        ],
        'CPCE' => [
            'code'  => 'CPCE',
            'name'  => 'Canada Post Expedited Parcel',
            'url'   => 'https://www.canadapost.ca/trackweb/en#/search',
            'query' => '?searchFor='
        ],
        'CPCP' => [
            'code'  => 'CPCP',
            'name'  => 'Canada Post Priority Parcel',
            'url'   => 'https://www.canadapost.ca/trackweb/en#/search',
            'query' => '?searchFor='
        ],
        'CPCR' => [
            'code'  => 'CPCR',
            'name'  => 'Canada Post',
            'url'   => 'https://www.canadapost.ca/trackweb/en#/search',
            'query' => '?searchFor='
        ],
        'CPCX' => [
            'code'  => 'CPCX',
            'name'  => 'Canada Post Xpresspost',
            'url'   => 'https://www.canadapost.ca/trackweb/en#/search',
            'query' => '?searchFor='
        ],
        'DHL' => [
            'code'  => 'DHL',
            'name'  => 'DHL',
            'url'   => 'https://www.dhl.com/',
            'query' => 'us-en/home/tracking.html?tracking-id='
        ],
        'DPDG' => [
            'code'  => 'DPDG',
            'name'  => 'DPD Group Ground',
            'url'   => 'https://www.dpdgroup.com/',
            'query' => 'nl/mydpd/my-parcels/track?parcelNumber='
        ],
        'EMAIL' => [
            'code'  => 'EMAIL',
            'name'  => 'Email Codes',
            'url'   => null,
            'query' => null
        ],
        'FD2' => [
            'code'  => 'FD2',
            'name'  => 'FedEx 2Day',
            'url'   => 'https://www.fedex.com/',
            'query' => 'fedextrack/?trknbr='
        ],
        'FDG' => [
            'code'  => 'FDG',
            'name'  => 'FedEx',
            'url'   => 'https://www.fedex.com/',
            'query' => 'fedextrack/?trknbr='
        ],
        'FDIE' => [
            'code'  => 'FDIE',
            'name'  => 'FedEx International Economy',
            'url'   => 'https://www.fedex.com/',
            'query' => 'fedextrack/?trknbr='
        ],
        'FDIP' => [
            'code'  => 'FDIP',
            'name'  => 'FedEx International Priority',
            'url'   => 'https://www.fedex.com/',
            'query' => 'fedextrack/?trknbr='
        ],
        'FDO' => [
            'code'  => 'FDO',
            'name'  => 'FedEx Standard Overnight',
            'url'   => 'https://www.fedex.com/',
            'query' => 'fedextrack/?trknbr='
        ],
        'FDPO' => [
            'code'  => 'FDPO',
            'name'  => 'FedEx Priority Overnight',
            'url'   => 'https://www.fedex.com/',
            'query' => 'fedextrack/?trknbr='
        ],
        'IGS' => [
            'code'  => 'IGS',
            'name'  => 'IGS Logistics Group Schreiner Pallets',
            'url'   => 'https://www.igslogistics.com',
            'query' => null
        ],
        'KN' => [
            'code'  => 'KN',
            'name'  => 'Kuhne & Nagel Road',
            'url'   => null,
            'query' => null
        ],
        'LOCAL' => [
            'code'  => 'LOCAL',
            'name'  => 'Local',
            'url'   => null,
            'query' => null
        ],
        'PCF' => [
            'code'  => 'PCF',
            'name'  => 'Parcelforce Express48',
            'url'   => 'https://www.parcelforce.com/track-trace',
            'query' => null
        ],
        'POTL' => [
            'code'  => 'POTL',
            'name'  => 'Poste Italiane Letter Mail',
            'url'   => null,
            'query' => null
        ],
        'PURP' => [
            'code'  => 'PURP',
            'name'  => 'Purolator',
            'url'   => 'https://www.purolator.com/en/shipping/tracker',
            'query' => '?pins='
        ],
        'RMFC' => [
            'code'  => 'RMFC',
            'name'  => 'Royal Mail First Class',
            'url'   => 'http://www.royalmail.com/portal/rm/track',
            'query' => '?trackNumber='
        ],
        'UPS1' => [
            'code'  => 'UPS1',
            'name'  => 'UPS Next Day Air',
            'url'   => 'https://www.ups.com/WebTracking',
            'query' => '?tracknum='
        ],
        'UPS2' => [
            'code'  => 'UPS2',
            'name'  => 'UPS 2nd Day Air',
            'url'   => 'https://www.ups.com/WebTracking',
            'query' => '?tracknum='
        ],
        'UPS3' => [
            'code'  => 'UPS3',
            'name'  => 'UPS 3 Day Air',
            'url'   => 'https://www.ups.com/WebTracking',
            'query' => '?tracknum='
        ],
        'UPSE' => [
            'code'  => 'UPSE',
            'name'  => 'UPS Worldwide Express',
            'url'   => 'https://www.ups.com/WebTracking',
            'query' => '?tracknum='
        ],
        'UPSG' => [
            'code'  => 'UPSG',
            'name'  => 'UPS',
            'url'   => 'https://www.ups.com/WebTracking',
            'query' => '?tracknum='
        ],
        'USPP' => [
            'code'  => 'USPP',
            'name'  => 'USPS Priority Mail',
            'url'   => 'https://tools.usps.com/go/TrackConfirmAction',
            'query' => '?tLabels='
        ],
        'USPS' => [
            'code'  => 'USPS',
            'name'  => 'USPS',
            'url'   => 'https://tools.usps.com/go/TrackConfirmAction',
            'query' => '?tLabels='
        ],
    ],
];
