<?php

namespace Ignite\Vendor\Helix\Laravel\Sso;

use Ignite\Vendor\Helix\Laravel\Contracts\DecoratableSsoClientInterface;
use Ignite\Vendor\Helix\Laravel\Contracts\ProvidesHelixPayload;
use Ignite\Vendor\Helix\Laravel\Exceptions\PayloadValidationFailed;
use Ignite\Vendor\Helix\Laravel\Exceptions\SsoBadLoginUrl;
use Ignite\Vendor\Helix\Laravel\Exceptions\SsoRequestFailed;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstFailedSsoRequest;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstMissingLoginUrl;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\RetrievesHelixUserIdAttribute;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\RetrievesSsoSessionKey;
use Ignite\Vendor\Helix\Laravel\Response;
use Illuminate\Http\Request;

class Session
{
    use RetrievesHelixUserIdAttribute,
        GuardAgainstFailedSsoRequest,
        GuardAgainstMissingLoginUrl,
        RetrievesSsoSessionKey;

    /**
     * @var Client
     */
    private $client;

    /**
     * Create the event listener.
     *
     * @param DecoratableSsoClientInterface $client
     */
    public function __construct(DecoratableSsoClientInterface $client)
    {
        $this->client = $client;
    }

    /**
     * Handle the Single Sign On session request for the given user.
     *
     * @param Request $request
     * @param ProvidesHelixPayload $user
     *
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function handle(Request $request, ProvidesHelixPayload $user)
    {
        $sessionKey = $this->getSessionKey();

        try {
            $helixIdAttribute = $this->getHelixUserIdAttribute();

            /** @var Payload $payload */
            $payload = $user->payload();

            /** @var Response $response */
            if ($user->getAttribute($helixIdAttribute)) {
                $response = $this->client->signIn($payload);
            } else {
                $response = $this->client->signUp($payload);
            }

            $this->guardAgainstFailedSsoRequest($payload, $response);
            $this->guardAgainstMissingLoginUrl($response);

            // Side-effect?
            if ($response->has(Response::KEY_HELIX_ID)) {
                $this->logIfNewHelixId($user, $response->get(Response::KEY_HELIX_ID));

                $user->forceFill([
                    $helixIdAttribute => $response->get(Response::KEY_HELIX_ID),
                ])->save();
            }

            // Side-effect
            $request->session()->put($sessionKey, $response->get(Response::KEY_SSO_URL));
        } catch (PayloadValidationFailed $exception) {
            $request->session()->put($sessionKey, Client::SSO_INVALID_DATA);
            $this->logException($exception, array_merge([
                'errors' => $exception->getErrors(),
                'request' => $request->except('password'),
                'user' => $user->only(['id', 'helix_user_id', 'email', 'first', 'last']),
            ]));
        } catch (SsoRequestFailed $exception) {
            $request->session()->put($sessionKey, Client::SSO_INVALID_RESPONSE);
            $this->logException($exception, array_merge(compact('request', 'user'), [
                'payload' => $exception->getPayload(),
                'response' => $exception->getResponse(),
            ]));
        } catch (SsoBadLoginUrl $exception) {
            $request->session()->put($sessionKey, Client::SSO_INVALID_LOGIN_URL);
            $this->logException($exception, array_merge(compact('request', 'user'), [
                'response' => $exception->getResponse(),
            ]));
        } catch (\Exception $exception) {
            $request->session()->put($sessionKey, Client::SSO_UNKNOWN_ERROR);
            $this->logException($exception, compact('request', 'user'));
        }
    }

    /**
     * Log the exception with as much data as possible.
     *
     * @param \Exception $exception
     * @param array $data
     */
    private function logException(\Exception $exception, array $data = [])
    {
        logger()->error($exception->getMessage(), array_merge(compact('exception'), $data));
    }

    /**
     * Log the new Helix ID if we were given a new one for the given user.
     *
     * @param $user
     * @param $newValue
     * @return void
     * @throws \Ignite\Vendor\Helix\Laravel\Exceptions\HelixUserIdAttributeNotConfigured
     */
    protected function logIfNewHelixId($user, $newValue): void
    {
        $helixIdAttribute = $this->getHelixUserIdAttribute();

        if ($user->{$helixIdAttribute} == $newValue) {
            return;
        }

        // Logging to a file whenever we get a new Helix user id. The hope is
        // when developing in beta, where the data in the database is a little
        // more volatile, we can recover those IDs without having to reach out
        // to Helix.
        $filePath = storage_path('app/helix-user-ids.csv');
        $needHeaders = ! file_exists($filePath);
        $fileHandle = fopen($filePath, 'w');

        if (! $fileHandle) {
            throw new \RuntimeException('Could not open '.$filePath.' for writing.');
        }

        if ($needHeaders) {
            fputcsv($fileHandle, ['member_group_id', 'catalog_id', 'client_user_id', 'helix_user_id']);
        }

        fputcsv($fileHandle, [
            config('helix.memberGroupId'),
            config('helix.catalog'),
            $user->getKey(),
            $newValue,
        ]);

        fclose($fileHandle);
    }
}
