<?php

namespace Ignite\Vendor\Helix\Laravel\Tests;

use Ignite\Vendor\Helix\Laravel\Sso\Payload;
use Ignite\Vendor\Helix\Laravel\Tests\Stubs\User;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Mockery;
use Orchestra\Testbench\TestCase as OrchestraTestCase;

class TestCase extends OrchestraTestCase
{
    /**
     * Setup the test environment.
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->migrateUserTable();
    }

    public static function setUpBeforeClass(): void
    {
        parent::setUpBeforeClass();

        static::publishConfig();
    }

    /**
     * Clean up the testing environment before the next test.
     *
     * @return void
     */
    protected function tearDown(): void
    {
        $this->deleteUserTable();

        parent::tearDown();
    }

    public static function tearDownAfterClass(): void
    {
        static::unpublishConfig();
    }

    /**
     * REgister the package service providers.
     *
     * @param \Illuminate\Foundation\Application $app
     *
     * @return array
     */
    protected function getPackageProviders($app)
    {
        return [
            \Ignite\Vendor\Helix\Laravel\HelixServiceProvider::class
        ];
    }

    /**
     * Register the package aliases
     * @param \Illuminate\Foundation\Application $app
     *
     * @return array
     */
    protected function getPackageAliases($app)
    {
        return [
            'Helix' => \Ignite\Vendor\Helix\Laravel\Facades\Helix::class,
        ];
    }

    /**
     * Define environment setup.
     *
     * @param \Illuminate\Foundation\Application $app
     *
     * @return void
     */
    protected function getEnvironmentSetUp($app)
    {
        $app['config']->set('helix.user.model', \Ignite\Vendor\Helix\Laravel\Tests\Stubs\User::class);
        $app['config']->set('database.default', 'testbench');
        $app['config']->set('database.connections.testbench', [
            'driver'   => 'sqlite',
            'database' => ':memory:',
            'prefix'   => '',
        ]);
    }

    /**
     * Create the user database table.
     *
     * @return void
     */
    protected function migrateUserTable()
    {
        Schema::create('users', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('first');
            $table->string('last');
            $table->string('role')->nullable()->default('participant');
            $table->string('email')->unique();
            $table->timestamp('email_verified_at')->nullable();
            $table->string('password');
            $table->string('helix_user_id')->nullable();
            $table->string('points');
            $table->string('address_1');
            $table->string('address_2');
            $table->string('city');
            $table->string('state');
            $table->string('postcode');
            $table->string('country');
            $table->string('phone');
            $table->string('locale')->nullable()->default('en_US');
            $table->rememberToken();
            $table->timestamps();
        });
    }

    /**
     * Delete the user database table.
     *
     * @return void
     */
    private function deleteUserTable()
    {
        Schema::drop('users');
    }

    /**
     * The default user data.
     *
     * @param array $overrides
     *
     * @return array
     */
    protected function getDefaultUserData($overrides = [])
    {
        return array_merge([
            'first' => 'John',
            'last' => 'Smith',
            'role' => 'participant',
            'helix_user_id' => null,
            'email' => 'it@brightspot.email',
            'email_verified_at' => now(),
            'password' => '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', // password
            'remember_token' => null,
            'points' => 1000,
            'address_1' => '102 Decker Ct',
            'address_2' => 'Suite 150',
            'city' => 'Irving',
            'state' => 'TX',
            'postcode' => '75065',
            'country' => 'US',
            'phone' => '9726616031',
            'locale' => 'en_US',
        ], $overrides);
    }

    /**
     * The Response data for a single sign on request.
     *
     * @param array $overrides
     *
     * @return array
     */
    protected function getResponseData($overrides = [])
    {
        $data = [
            'status' => "success",
        ];

        if (! isset($overrides['code']) || $overrides['code'] === 200) {
            $data = array_merge($data, [
                'code' => 200,
                'catalogId' => "A14E715F-CAF0-4749-B569-58059C9DD490",
                'clientUserId' => "1",
                'helixUserId' => 690901
            ]);
        }

        return array_merge($data, $overrides);
    }

    /**
     * The payload data.
     *
     * @param array $overrides
     *
     * @return array
     */
    protected function getPayloadData($overrides = [])
    {
        return array_merge([
            //'helixUserId' => 690901,
            'clientUserId' => 1,
            'catalogId' => "A14E715F-CAF0-4749-B569-58059C9DD490",
            'memberGroupId' => null,
            'accessToken' => "E7DFA4373E531A7AA23B1955B94E6526F4B33587FFE12A758DCB35E48736251B",
            'firstName' => 'John',
            'lastName' => 'Smith',
            'shipAddress1' => '102 Decker Ct',
            'shipAddress2' => 'Suite 150',
            'shipCity' => 'Irving',
            'shipState' => 'TX',
            'shipPostalCode' => '75065',
            'shipCountry' => 'US',
            'phoneNumber1' => '9726616031',
            'emailAddress' => 'it@brightspot.email'
        ], $overrides);
    }

    /**
     * The mocked user model.
     *
     * @param Payload $payload
     *
     * @return User|Mockery\MockInterface
     */
    protected function getMockUser($payload)
    {
        $user = Mockery::mock(User::class);
        $user->shouldReceive('payload')->once()->andReturn($payload);

        return $user;
    }

    /**
     * Publish the config file after all tests complete.
     *
     * @return void
     */
    protected static function publishConfig()
    {
        copy(
            realpath(__DIR__ . '/../config/helix.php'),
            __DIR__ . '/../vendor/orchestra/testbench-core/laravel/config/helix.php'
        );
    }

    /**
     * Unpublish the config file after all tests complete.
     *
     * @return void
     */
    protected static function unpublishConfig()
    {
        $path = realpath(__DIR__ . '/../vendor/orchestra/testbench-core/laravel/config/helix.php');

        if ($path && file_exists($path)) {
            unlink($path);
        }
    }
}
