<?php

namespace Ignite\Vendor\Helix\Laravel\Sso;

use Ignite\Vendor\Helix\Laravel\Contracts\DecoratableSsoClientInterface;
use Ignite\Vendor\Helix\Laravel\Contracts\ProvidesHelixPayload;
use Ignite\Vendor\Helix\Laravel\Exceptions\PayloadValidationFailed;
use Ignite\Vendor\Helix\Laravel\Exceptions\SsoBadLoginUrl;
use Ignite\Vendor\Helix\Laravel\Exceptions\SsoRequestFailed;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstFailedSsoRequest;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstMissingLoginUrl;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\RetrievesHelixUserIdAttribute;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\RetrievesSsoSessionKey;
use Ignite\Vendor\Helix\Laravel\Response;
use Illuminate\Http\Request;

class Session
{
    use RetrievesHelixUserIdAttribute,
        GuardAgainstFailedSsoRequest,
        GuardAgainstMissingLoginUrl,
        RetrievesSsoSessionKey;

    /**
     * @var Client
     */
    private $client;

    /**
     * Create the event listener.
     *
     * @param DecoratableSsoClientInterface $client
     */
    public function __construct(DecoratableSsoClientInterface $client)
    {
        $this->client = $client;
    }

    /**
     * Handle the Single Sign On session request for the given user.
     *
     * @param Request $request
     * @param ProvidesHelixPayload $user
     *
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function handle(Request $request, ProvidesHelixPayload $user)
    {
        $sessionKey = $this->getSessionKey();

        try {
            $helixIdAttribute = $this->getHelixUserIdAttribute();

            /** @var Payload $payload */
            $payload = $user->payload();

            /** @var Response $response */
            if ($user->getAttribute($helixIdAttribute)) {
                $response = $this->client->signIn($payload);
            } else {
                $response = $this->client->signUp($payload);
            }

            $this->guardAgainstFailedSsoRequest($payload, $response);
            $this->guardAgainstMissingLoginUrl($response);

            // Side-effect?
            if ($response->has(Response::KEY_HELIX_ID)) {
                $user->forceFill([
                    $helixIdAttribute => $response->get(Response::KEY_HELIX_ID)
                ])->save();
            }

            // Side-effect
            $request->session()->put($sessionKey, $response->get(Response::KEY_SSO_URL));
        } catch (PayloadValidationFailed $exception) {
            $request->session()->put($sessionKey, Client::SSO_INVALID_DATA);
            $this->logException($exception, array_merge([
                'errors' => $exception->getErrors(),
                'request' => $request->except('password'),
                'user' => $user->only(['id', 'helix_user_id', 'email', 'first', 'last'])
            ]));
        } catch (SsoRequestFailed $exception) {
            $request->session()->put($sessionKey, Client::SSO_INVALID_RESPONSE);
            $this->logException($exception, array_merge(compact('request', 'user'), [
                'payload' => $exception->getPayload(),
                'response' => $exception->getResponse()
            ]));
        } catch (SsoBadLoginUrl $exception) {
            $request->session()->put($sessionKey, Client::SSO_INVALID_LOGIN_URL);
            $this->logException($exception, array_merge(compact('request', 'user'), [
                'response' => $exception->getResponse()
            ]));
        } catch (\Exception $exception) {
            $request->session()->put($sessionKey, Client::SSO_UNKNOWN_ERROR);
            $this->logException($exception, compact('request', 'user'));
        }
    }

    /**
     * Log the exception with as much data as possible.
     *
     * @param \Exception $exception
     * @param array $data
     */
    private function logException(\Exception $exception, array $data = [])
    {
        logger()->error($exception->getMessage(), array_merge(compact('exception'), $data));
    }
}
