<?php

namespace Ignite\Vendor\Helix\Laravel\Tests\Feature\Http\Controllers;

use Ignite\Vendor\Helix\Laravel\Events\WebhookResourceCreated;
use Ignite\Vendor\Helix\Laravel\Response as HelixResponse;
use Ignite\Vendor\Helix\Laravel\Services\Token;
use Ignite\Vendor\Helix\Laravel\Tests\TestCase;
use Ignite\Vendor\Helix\Laravel\Tests\Stubs\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Event;

class WebhookControllerTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_can_return_a_200_when_the_webhook_is_processed()
    {
        Event::fake();

        User::create($this->getDefaultUserData());

        $response = $this->postJson(route('api.helix.webhook'), [
                'clientUserId' => 1,
                'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 1),
                'type' => 'order',
	            'orderNumber' => '10131001',
            ]);

        $response->assertStatus(Response::HTTP_OK);

        $this->assertEquals(Response::HTTP_OK, $response->json('code'));
        $this->assertEquals(trans('helix::messages.webhook.success'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_SUCCESS, $response->json('status'));

        Event::assertDispatched(WebhookResourceCreated::class);
    }

    /**
     * @test
     */
    public function it_will_return_a_403_when_the_access_code_is_invalid()
    {
        Event::fake();

        $response = $this->postJson(route('api.helix.webhook'), [
            'clientUserId' => 1,
            'accessToken' => 'foo',
            'type' => 'order',
            'orderNumber' => '10131001',
        ]);

        $response->assertStatus(Response::HTTP_FORBIDDEN);

        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->json('code'));
        $this->assertEquals(trans('helix::messages.webhook.forbidden'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));

        Event::assertNotDispatched(WebhookResourceCreated::class);
    }

    /**
     * @test
     */
    public function it_will_return_a_403_when_the_access_code_is_formatted_correctly_but_for_the_wrong_user()
    {
        Event::fake();

        $response = $this->postJson(route('api.helix.webhook'), [
            'clientUserId' => 1,
            'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 2),
            'type' => 'order',
            'orderNumber' => '10131001',
        ]);

        $response->assertStatus(Response::HTTP_FORBIDDEN);

        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->json('code'));
        $this->assertEquals(trans('helix::messages.webhook.forbidden'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));

        Event::assertNotDispatched(WebhookResourceCreated::class);
    }

    /**
     * @test
     */
    public function it_will_return_a_503_when_an_unknown_error_occurs()
    {
        Event::fake();
        User::create($this->getDefaultUserData());

        $response = $this->postJson(route('api.helix.webhook'), [
            'clientUserId' => [], // Pass an array to force an exception
            'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 1),
            'type' => 'order',
            'orderNumber' => '10131001',
        ]);

        $response->assertStatus(Response::HTTP_SERVICE_UNAVAILABLE);

        $this->assertEquals(Response::HTTP_SERVICE_UNAVAILABLE, $response->json('code'));
        $this->assertEquals(trans('helix::messages.webhook.unavailable'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));

        Event::assertNotDispatched(WebhookResourceCreated::class);
    }
}
