<?php

namespace Ignite\Vendor\Helix\Laravel\Tests\Feature\Http\Controllers;

use Ignite\Vendor\Helix\Laravel\Events\OrderResourceCreated;
use Ignite\Vendor\Helix\Laravel\Events\WebhookResourceCreated;
use Ignite\Vendor\Helix\Laravel\Response as HelixResponse;
use Ignite\Vendor\Helix\Laravel\Services\Token;
use Ignite\Vendor\Helix\Laravel\Tests\TestCase;
use Ignite\Vendor\Helix\Laravel\Tests\Stubs\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Event;

class OrderControllerTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_can_return_a_200_when_the_order_is_processed()
    {
        Event::fake();

        User::create($this->getDefaultUserData());

        $response = $this->postJson(route('api.helix.order'), $this->getOrderData());

        $response->assertStatus(Response::HTTP_OK);

        $this->assertEquals(Response::HTTP_OK, $response->json('code'));
        $this->assertEquals(trans('helix::messages.order.success'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_SUCCESS, $response->json('status'));

        Event::assertDispatched(OrderResourceCreated::class);
    }

    /**
     * @test
     */
    public function it_will_return_a_403_when_the_access_code_is_invalid()
    {
        Event::fake();

        $response = $this->postJson(route('api.helix.order'), $this->getOrderData([
            'accessToken' => 'foo',
        ]));

        $response->assertStatus(Response::HTTP_FORBIDDEN);

        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->json('code'));
        $this->assertEquals(trans('helix::messages.order.forbidden'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));

        Event::assertNotDispatched(OrderResourceCreated::class);
    }

    /**
     * @test
     */
    public function it_will_return_a_403_when_the_access_code_is_formatted_correctly_but_for_the_wrong_user()
    {
        Event::fake();

        $response = $this->postJson(route('api.helix.order'), $this->getOrderData([
            'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 2),
        ]));

        $response->assertStatus(Response::HTTP_FORBIDDEN);

        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->json('code'));
        $this->assertEquals(trans('helix::messages.order.forbidden'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));

        Event::assertNotDispatched(OrderResourceCreated::class);
    }

    /**
     * @test
     */
    public function it_will_return_a_503_when_an_unknown_error_occurs()
    {
        Event::fake();
        User::create($this->getDefaultUserData());

        $response = $this->postJson(route('api.helix.order'), $this->getOrderData([
            'productCount' => [], // Pass an array to force an exception
        ]));

        $response->assertStatus(Response::HTTP_SERVICE_UNAVAILABLE);

        $this->assertEquals(Response::HTTP_SERVICE_UNAVAILABLE, $response->json('code'));
        $this->assertEquals(trans('helix::messages.order.unavailable'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));

        Event::assertNotDispatched(OrderResourceCreated::class);
    }

    /**
     * Get the order
     * @param array $overrides
     *
     * @return array
     */
    private function getOrderData($overrides = [])
    {
        return array_merge([
            'clientUserId' => 1,
            'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 1),
            'orderNumber' => '101000131',
            'firstName' => 'John',
            'lastName' => 'Smith',
            'shipAddress1' => '102 Decker Ct',
            'shipAddress2' => 'Suite 150',
            'shipCity' => 'Irving',
            'shipState' => 'TX',
            'shipPostalCode' => '75062',
            'shipCountry' => 'US',
            'phoneNumber' => '555-123-4567',
            'productCount' => '2',
            'pointsToDeduct' => '750',
            'products' => [
                [
                    'productId' => 752,
                    'productName' => '$50 Golf Town Gift Card',
                    'pointPerItem' => 1,
                    'quantity' => 1,
                    'subPoints' => 50
                ],
                [
                    'productId' => 754,
                    'productName' => '$50 Amazon Gift Card',
                    'pointPerItem' => 1,
                    'quantity' => 1,
                    'subPoints' => 50
                ]
            ]
        ], $overrides);
    }
}
