<?php

namespace Ignite\Vendor\Helix\Laravel;

use Ignite\Vendor\Helix\Adapters\Guzzle;
use Ignite\Vendor\Helix\Authentication;
use Ignite\Vendor\Helix\Decorators\Json;
use Ignite\Vendor\Helix\Laravel\Carriers\ArrayCollection;
use Ignite\Vendor\Helix\Laravel\Carriers\CarrierCollectionInterface;
use Ignite\Vendor\Helix\Laravel\Commands\DecryptAccessToken;
use Ignite\Vendor\Helix\Laravel\Commands\EncryptAccessToken;
use Ignite\Vendor\Helix\Factory;
use Ignite\Vendor\Helix\HelixInterface;
use Ignite\Vendor\Helix\Laravel\Commands\PurgeProcessedResources;
use Ignite\Vendor\Helix\Laravel\Contracts\DecoratableSsoClientInterface;
use Ignite\Vendor\Helix\Laravel\Http\Views\Composers\IframeComposer;
use Ignite\Vendor\Helix\Laravel\Http\Views\Composers\ScriptComposer;
use Ignite\Vendor\Helix\Laravel\Services\Token;
use Ignite\Vendor\Helix\Laravel\Sso\Client;
use Illuminate\Auth\Events\Login;
use Illuminate\Auth\Events\Registered;
use Illuminate\Database\Eloquent\Factory as LaravelDatabaseFactory;
use Illuminate\Foundation\Application;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\View;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Str;

class HelixServiceProvider extends ServiceProvider
{
    /**
     * When Laravel is booting up, register configuration.
     */
    public function boot()
    {
        $this->loadRoutesFrom($this->helixPath('routes/api.php'));

        $this->loadMigrationsFrom($this->helixPath('database/migrations'));

        $this->loadViewsFrom($this->helixPath('resources/views'), 'helix');

        $this->loadTranslationsFrom($this->helixPath('resources/lang'), 'helix');

        $this->mergeConfigFrom($this->helixConfigPath(), 'helix');

        $this->publishes([
            $this->helixConfigPath() => config_path('helix.php'),
            $this->helixPath('resources/views') => resource_path('views/vendor/helix'),
            $this->helixPath('resources/lang') => resource_path('lang/vendor/helix'),
        ]);

        if ($this->app->runningInConsole()) {
            $this->commands([
                DecryptAccessToken::class,
                EncryptAccessToken::class,
                PurgeProcessedResources::class,
            ]);
        }

        View::composer('helix::catalog.iframe', IframeComposer::class);
        View::composer('helix::catalog.script', ScriptComposer::class);

        foreach (array_unique($this->app['config']->get('helix.sso.listeners.login', [])) as $listener) {
            Event::listen(Login::class, $listener);
        }

        foreach (array_unique($this->app['config']->get('helix.sso.listeners.registered', [])) as $listener) {
            Event::listen(Registered::class, $listener);
        }

        if ($this->app->isLocal() && $this->app->runningInConsole()) {
            $this->app->make(LaravelDatabaseFactory::class)->load($this->helixPath('database/factories'));
        }
    }

    /**
     * When Laravel is registering service providers, register the ClientInterface
     * and provide an alias for use with a Facade.
     */
    public function register()
    {
        $this->app->singleton(Token::class, $this->closure('registerTokenService'));

        $this->app->singleton(HelixInterface::class, $this->closure('registerHelix'));

        $this->app->singleton(DecoratableSsoClientInterface::class, $this->closure('registerSsoClient'));

        $this->app->singleton(CarrierCollectionInterface::class, $this->closure('registerCarrierCollection'));

        $this->app->alias(HelixInterface::class, 'helix');
    }

    /**
     * Register the Helix client customized for Laravel.
     *
     * @param Application $app
     *
     * @return \Ignite\Vendor\Helix\DecoratorInterface
     */
    public function registerHelix($app)
    {
        $config = $app['config']->get('helix.api');

        $factory = new Factory($config['adapters'], $config['decorators']);

        return $factory->make(
            $config['username'],
            $config['password'],
            $config['endpoint'],
            $config['adapter'],
            $config['decorator']
        );
    }

    /**
     * Register the Access Token Encryption and Decryption Service.
     *
     * @param Application $app
     *
     * @return Token
     * @throws \Exception
     */
    public function registerTokenService($app)
    {
        return new Token(
            $app['config']->get('helix.token.key'),
            $app['config']->get('helix.token.cipher')
        );
    }

    /**
     * Register the Single Sign On client instance.
     *
     * @param Application $app
     *
     * @return Client
     */
    public function registerSsoClient($app)
    {
        $config = $app['config'];

        // Get the configured adapters so we can look up the shortname.
        $adapters = $config->get('helix.sso.adapters', []);
        $adapterClass = $adapters[$config->get('helix.sso.adapter')] ?? Guzzle::class;

        // Get the configured decorators so we can look up the shortname.
        $decorators = $config->get('helix.sso.decorators', []);
        $decoratorClass = $decorators[$config->get('helix.sso.decorator')] ?? Json::class;

        // Instantiate the Authentication class.
        $authentication = new Authentication(
            $config->get('helix.sso.username'),
            $config->get('helix.sso.password'),
            $config->get('helix.sso.endpoint')
        );

        // Instantiate the configured adapter.
        $adapter = new $adapterClass($authentication);

        // Instantiate the client.
        $client = new Client($adapter);

        // Decorate the client
        return new $decoratorClass($client);
    }

    /**
     * Register the Carrier Collection instance.
     *
     * @param Application $app
     *
     * @return CarrierCollectionInterface
     */
    public function registerCarrierCollection($app)
    {
        return new ArrayCollection($app['config']->get('helix.carriers'));
    }

    /**
     * Create a Closure from a callable.
     *
     * @param string $method
     *
     * @return \Closure
     */
    protected function closure(string $method)
    {
        return \Closure::fromCallable([$this, $method]);
    }

    /**
     * The path to the config file.
     *
     * @return bool|string
     */
    protected function helixConfigPath()
    {
        return $this->helixPath("/config/helix.php");
    }

    /**
     * The path to the helix bridge directory.
     *
     * @param string $path
     *
     * @return bool|string
     */
    protected function helixPath($path = '/')
    {
        $path = Str::start(str_replace(['\\', '//'], '/', (string) $path), '/');

        return realpath(__DIR__ . '/..' . $path);
    }
}
