# Helix Laravel Bridge

## Installation

```
composer require ignite/helix-api-laravel
```

## Configuration

### Publishing the configuration file

```
php artisan vendor:publish --force --provider=Ignite\Vendor\Helix\Laravel\HelixServiceProvider
```

### Making the routes work:

Open up your `VerifyCsrfToken` Middleware file located at `app/Http/Middleware/VerifyCsrfToken.php` and add the following exceptions:

```
    protected $except = [
        '/api/helix/points',
        '/api/helix/order',
        '/api/helix/webhooks'
    ];
```

### Write an Event Listener for when an Order is created.

```
php artisan make:listener StoreOrderDetails --event=\\Ignite\\Vendor\\Helix\\Laravel\\Events\\OrderResourceCreated
```

The order resource contains the following information:

```
'order_number' => nullable string
'user_id' => unsigned big integer
'access_token' => string
'qty' => unsigned big integer
'points' => unsigned big integer containing the points to deduct
'first' => string
'last' => string
'ship_address_1' => string
'ship_address_2' => string
'ship_city' => string
'ship_state' => string
'ship_postcode' => string
'ship_country' => string
'ship_phone' => string
'products' => json
'status' => nullable string, default value: 'processing'
```

Once you 'process' the order resource, you should mark the `status` as `processed`.

## Single Sign On

### Writing a Payload transformer for SSO

Create a file called `PayloadTransformer` in `app/Transformers`.

```php
<?php

namespace App\Transformers;

use App\User;
use Ignite\Vendor\Helix\Laravel\Services\Token;
use Illuminate\Config\Repository;

class PayloadTransformer
{
    /**
     * @var Token
     */
    private $tokenService;

    /**
     * @var Repository
     */
    private $config;

    /**
     * PayloadTransformer constructor.
     *
     * @param Token $tokenService
     * @param Repository $config
     */
    public function __construct(Token $tokenService, Repository $config)
    {
        $this->tokenService = $tokenService;
        $this->config = $config;
    }

    /**
     * Transform the user for use with Helix.
     *
     * @param User $user
     *
     * @return array
     */
    public function transform(User $user)
    {
        return [
            'helixUserId' => $user->helixUserId,
            'catalogId' => 'A14E715F-CAF0-4749-B569-58059C9DD490',
            'accessToken' => $this->getAccessToken($user),
            'clientUserId' => $user->getKey(),
            'firstName' => $user->first,
            'lastName' => $user->last,
            'primaryLanguage' => $this->getPrimaryLanguage($user),
            'points' => $user->points,
            'phoneNumber1' => $user->phone,
            'emailAddress' => $user->email,
            'shipAddress1' => $user->address_1,
            'shipAddress2' => $user->address_2,
            'shipCity' => $user->city,
            'shipState' => $user->state,
            'shipPostalCode' => $user->postcode,
            'shipCountry' => $user->country,
        ];
    }

    /**
     * Get the encrypted access token for the given user.
     *
     * @param User $user
     *
     * @return string
     */
    private function getAccessToken(User $user)
    {
        return $this->tokenService->encrypt($this->config->get('helix.token.catalog'), $user->getKey());
    }

    /**
     * Get the user's configured primary language.
     *
     * @param User $user
     *
     * @return bool|string
     */
    private function getPrimaryLanguage(User $user)
    {
        if (empty($user->locale)) {
            return 'en';
        }

        return substr($user->locale, 0, 2);
    }
}

```

You can call the PayloadTransformer on your `User` model like so:

```php
<?php

namespace App;

use App\Transformers\PayloadTransformer;
use Ignite\Vendor\Helix\Laravel\Contracts\ProvidesHelixPayload;
use Ignite\Vendor\Helix\Laravel\Resources\Concerns\Payloadable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Foundation\Auth\User as Authenticatable;

class User extends Authenticatable implements ProvidesHelixPayload
{
    use Notifiable, Payloadable;

    /**
     * Provide the helix data necessary to authenticate a user.
     *
     * @return array
     */
    public function toPayload()
    {
        return app(PayloadTransformer::class)->transform($this);
    }
}

```
