<?php

return [
    /**
     * The user information configured here defines the model and attributes to be used by
     * the HTTP routes and event listeners in this package in order to determine the
     * user model to use to query for helix specific data in an abstract way.
     */
    'user' => [
        /**
         * The user model to query when verifying points and authorizing a user with
         * Helix. The model value should be the fully qualified class name of an
         * eloquent class that implements the ProvidesHelixPayload interface
         * and the Payloadable trait to transform user data for Helix.
         *
         * @see \Ignite\Vendor\Helix\Laravel\Contracts\ProvidesHelixPayload
         * @see \Ignite\Vendor\Helix\Laravel\Resources\Concerns\Payloadable
         * @see \Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\RetrievesUserModel
         */
        'model' => \App\User::class,

        /**
         * The following attributes help the package to identify the data attributes
         * required to match user id values and determine the number of points
         * for a user/participant and verify the provided access token.
         *
         * @see \Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\RetrievesUserIdAttribute
         * @see \Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\RetrievesPointsAttribute
         * @see \Ignite\Vendor\Helix\Laravel\Listeners\Concerns\RetrievesHelixUserIdAttribute
         */
        'attributes' => [
            'id' => 'id',
            'points' => 'points',
            'helix_user_id' => 'helix_user_id'
        ]
    ],

    /**
     * The following configuration defines the shared token values for use in a Helix
     * access token to be encrypted and decrypted securely. The Token service will
     * verify that the cipher is available on the server or throw an exception.
     *
     * @see \Ignite\Vendor\Helix\Laravel\Services\Token
     * @see \Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\GuardAgainstInvalidToken
     */
    'token' => [
        /**
         * A 16 character alpha-numeric randomg string. The shared catalog token is
         * typically defined by Helix during the client onboarding phase. When
         * setting up a new integration, we should contact Hassan Tariq.
         */
        'catalog' => env('HELIX_TOKEN_CATALOG'),

        /**
         * A cryptographically secure, base64 encoded shared key will be provided by
         * Helix and will be used to encrypt and decrypt the access token sent in
         * requests going to and coming from Helix. Keep it secret. Keep it safe.
         */
        'key' => env('HELIX_TOKEN_KEY', ''),

        /**
         * The encryption algorithm used is AES-128-ECB, this specific algorithm was
         * mandated by Helix. If the cipher type ever changes, we would also need
         * to generate compatible shared key of the correct number of bytes.
         */
        'cipher' => env('HELIX_TOKEN_CIPHER', 'AES-128-ECB'),
    ],

    /**
     * The Helix catalog ID is a UUID generated by Brightspot and sent to Helix in
     * order to identify the catalog between our two systems. The value will be
     * sent with every SSO request and will be included in many responses too.
     */
    'catalog' => env('HELIX_CATALOG_ID', ''),

    /**
     * The single sign on configuration defines the API endpoint, username and password
     * used when we attempt to sign into Helix from our application. We also define
     * the components to handle the request and response as well as session data
     * and event listeners which handle resolving the user payload during the
     * sign-in and sign-up processes.
     *
     * @see \Ignite\Vendor\Helix\Laravel\Sso\Client
     * @see \Ignite\Vendor\Helix\Laravel\Sso\Payload
     * @see \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnLogin
     * @see \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnRegister
     * @see \Ignite\Vendor\Helix\Laravel\Contracts\SsoClientInterface
     * @see \Ignite\Vendor\Helix\Laravel\Contracts\DecoratableSsoClientInterface
     * @see \Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstMissingLoginUrl
     * @see \Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstFailedSsoRequest
     * @see \Ignite\Vendor\Helix\Laravel\Exceptions\SsoRequestFailed
     * @see \Ignite\Vendor\Helix\Laravel\Exceptions\SsoBadLoginUrl
     */
    'sso' => [
        /**
         * The Helix single sign on endpoint URL is provided by Helix and should be
         * located at our custom subdomain for Brightspot. In order to find out
         * exactly what the value should be, we should contact Hassan Tariq.
         */
        'endpoint' => env('HELIX_SSO_ENDPOINT', ''),

        /**
         * The Helix single sign on username is provided by Helix. In order to find
         * out exactly what the value should be, we should contact Hassan Tariq.
         */
        'username' => env('HELIX_SSO_USERNAME', ''),

        /**
         * The Helix single sign on password is provided by Helix. In order to find
         * out exactly what the value should be, we should contact Hassan Tariq.
         */
        'password' => env('HELIX_SSO_PASSWORD', ''),

        /**
         * The Laravel session information required to store the single sign on login
         * URL consistently to the correct key. The event listeners will store the
         * value and the view composer will make it available for display.
         *
         * @see \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnLogin
         * @see \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnRegister
         * @see \Ignite\Vendor\Helix\Laravel\Http\Views\Composers\IframeComposer
         */
        'session' => [
            /**
             * The string to use as the key when storing single sign on information
             * in the session. Defining it here makes it easy for our application
             * to consume the key consistently through out the SSO lifecycle.
             */
            'key' => env('HELIX_SSO_SESSION_KEY', 'helix_login_url'),
        ],

        /**
         * The single sign on event listeners to be registered with the correct events
         * via the Helix service provider. The single sign on synchronization code
         * is triggered by the user login event and user registration event.
         *
         * @see \Ignite\Vendor\Helix\Laravel\HelixServiceProvider
         * @see \Illuminate\Auth\Events\Login
         * @see \Illuminate\Auth\Events\Registered
         */
        'listeners' => [
            /**
             * The user registered event is a Laravel core event, if you're not using
             * the core auth scaffolding, please make sure that you are dispatching
             * the registered event from your custom auth controller code.
             */
            'registered' => [
                \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnRegister::class,
            ],
            /**
             * The user login event is a Laravel core event, if you're not using the
             * core auth scaffolding, please make sure that you are dispatching
             * the login event from your custom auth controller code.
             */
            'login' => [
                \Ignite\Vendor\Helix\Laravel\Listeners\SyncOnLogin::class,
            ]
        ],

        /**
         * The Helix adapter to use. By default we use the Guzzle Adapter since it already
         * implements the PSR-7 Http Standard Interfaces. If Guzzle was ever deprecated
         * we could swap the implementation out very easily from within this project.
         */
        'adapter' => 'guzzle',

        /**
         * The Helix decorator to use. By default we use the JSON decorator, JSON is the
         * easiest data format to use with the Helix API. However, we could swap out
         * the implementation easily if we ever needed to transform the response.
         */
        'decorator' => 'json',

        /**
         * The default adapters can be changed by merging the array below. The list
         * of adapter definitions must be in a consistent key-value pair format,
         * for example: 'short_name' => \Fully\Qualified\Class\Name::class
         */
        'adapters' => [
            'guzzle' => \Ignite\Vendor\Helix\Adapters\Guzzle::class
        ],

        /**
         * The default decorators can be changed by merging the array below. The list
         * of decorators definitions must be in a consistent key-value pair format,
         * for example: 'short_name' => \Fully\Qualified\Class\Name::class
         */
        'decorators' => [
            'json' => \Ignite\Vendor\Helix\Laravel\Decorators\Laravel::class
        ]
    ],

    'api' => [
        /**
         * The Helix API endpoint URL is provided by Helix and should be located at
         * our custom subdomain for Brightspot but may be different to the SSO
         * endpoint. We should contact Hassan Tariq to ascertain this value.
         */
        'endpoint' => env('HELIX_API_ENDPOINT', ''),

        /**
         * The Helix API username is provided by Helix but may be different to the
         * SSO username. In order to find out exactly what the value should be,
         * we should contact Hassan Tariq.
         */
        'username' => env('HELIX_API_USERNAME', ''),

        /**
         * The Helix API password is provided by Helix but may be different to the
         * SSO password. In order to find out exactly what the value should be,
         * we should contact Hassan Tariq.
         */
        'password' => env('HELIX_API_PASSWORD', ''),

        /**
         * The Helix adapter to use. By default we use the Guzzle Adapter since it already
         * implements the PSR-7 Http Standard Interfaces. If Guzzle was ever deprecated
         * we could swap the implementation out very easily from within this project.
         */
        'adapter' => 'guzzle',

        /**
         * The Helix decorator to use. By default we use the JSON decorator, JSON is the
         * easiest data format to use with the Helix API. However, we could swap out
         * the implementation easily if we ever needed to transform the response.
         */
        'decorator' => 'json',

        /**
         * The default adapters can be changed by merging the array below. The list
         * of adapter definitions must be in a consistent key-value pair format,
         * for example: 'short_name' => \Fully\Qualified\Class\Name::class
         */
        'adapters' => [
            'guzzle' => \Ignite\Vendor\Helix\Adapters\Guzzle::class
        ],

        /**
         * The default decorators can be changed by merging the array below. The list
         * of decorators definitions must be in a consistent key-value pair format,
         * for example: 'short_name' => \Fully\Qualified\Class\Name::class
         */
        'decorators' => [
            'json' => \Ignite\Vendor\Helix\Laravel\Decorators\Laravel::class
        ]
    ]
];
