<?php

namespace Ignite\Vendor\Helix\Laravel\Http\Controllers;

use Ignite\Vendor\Helix\Laravel\Exceptions\InvalidAccessToken;
use Ignite\Vendor\Helix\Laravel\Events\OrderResourceCreated;
use Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\TranslatesMessages;
use Ignite\Vendor\Helix\Laravel\Resources\Order;
use Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\CommonResponses;
use Ignite\Vendor\Helix\Laravel\Http\Controllers\Concerns\GuardAgainstInvalidToken;
use Ignite\Vendor\Helix\Laravel\Services\Token;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;

class OrderController
{
    use GuardAgainstInvalidToken, CommonResponses, TranslatesMessages;

    /**
     * Create the order.
     *
     * @param Request $request
     * @param Token $tokenService
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function __invoke(Request $request, Token $tokenService)
    {
        try {
            $this->guardAgainstInvalidAccessToken($request, $tokenService);

            $orderResource = Order::updateOrCreate(['order_number' => $request->json()->get('orderNumber')], [
                'user_id' => $request->json()->get('clientUserId'),
                'access_token' => $request->json()->get('accessToken'),
                'order_number' => $request->json()->get('orderNumber'),
                'first' => $request->json()->get('firstName'),
                'last' => $request->json()->get('lastName'),
                'ship_address_1' => $request->json()->get('shipAddress1'),
                'ship_address_2' => $request->json()->get('shipAddress2'),
                'ship_city' => $request->json()->get('shipCity'),
                'ship_state' => $request->json()->get('shipState'),
                'ship_postcode' => $request->json()->get('shipPostalCode'),
                'ship_country' => $request->json()->get('shipCountry'),
                'ship_phone' => $request->json()->get('phoneNumber'),
                'qty' => $request->json()->get('qty', $request->json()->get('productCount', 0)),
                'points' => $request->json()->get('pointsToDeduct'),
                'products' => $request->json()->get('products', []),
                'extra' => Arr::except($request->json()->all(), [
                    'clientUserId',
                    'accessToken',
                    'orderNumber',
                    'firstName',
                    'lastName',
                    'shipAddress1',
                    'shipAddress2',
                    'shipCity',
                    'shipState',
                    'shipPostalCode',
                    'shipCountry',
                    'phoneNumber',
                    'productCount',
                    'pointsToDeduct',
                    'products',
                    'qty'
                ])
            ]);

            OrderResourceCreated::dispatch($orderResource);

            return $this->successResponse($request, $this->message('order.success'));
        } catch (InvalidAccessToken $exception) {
            logger()->error($exception->getMessage(), compact('exception', 'request'));

            return $this->forbiddenResponse($request, $this->message('order.forbidden'));
        } catch (\Exception $exception) {
            logger()->error($exception->getMessage(), compact('exception', 'request'));

            return $this->serviceUnavailableResponse($request, $this->message('order.unavailable'));
        }
    }
}
