<?php

namespace Ignite\Vendor\Helix\Laravel\Tests\Feature\Http\Controllers;

use Ignite\Vendor\Helix\Laravel\Response as HelixResponse;
use Ignite\Vendor\Helix\Laravel\Services\Token;
use Ignite\Vendor\Helix\Laravel\Tests\TestCase;
use Ignite\Vendor\Helix\Laravel\Tests\Stubs\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Response;

class PointsControllerTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_can_return_points_successfully()
    {
        $user = User::create($this->getDefaultUserData());

        $response = $this->withoutExceptionHandling()
            ->postJson(route('api.helix.points'), [
                'clientUserId' => '1',
                'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 1)
            ]);

        $response->assertStatus(Response::HTTP_OK);

        $this->assertEquals($user->points, $response->json('points'));
        $this->assertEquals(Response::HTTP_OK, $response->json('code'));
        $this->assertEquals(trans('helix::messages.points.success'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_SUCCESS, $response->json('status'));
    }

    /**
     * @test
     */
    public function it_will_return_a_404_when_user_not_found()
    {
        $response = $this->postJson(route('api.helix.points'), [
             'clientUserId' => '12',
             'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 12)
         ]);

        $response->assertStatus(Response::HTTP_NOT_FOUND);
        $this->assertEquals(Response::HTTP_NOT_FOUND, $response->json('code'));
        $this->assertEquals(trans('helix::messages.points.missing', ['user' => 12]), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));
    }

    /**
     * @test
     */
    public function it_will_return_a_403_when_the_access_code_is_invalid()
    {
        $response = $this->postJson(route('api.helix.points'), [
            'clientUserId' => '12',
            'accessToken' => 'foo'
        ]);

        $response->assertStatus(Response::HTTP_FORBIDDEN);
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->json('code'));
        $this->assertEquals(trans('helix::messages.points.forbidden'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));
    }

    /**
     * @test
     */
    public function it_will_return_a_403_when_the_access_code_is_formatted_correctly_but_for_the_wrong_user()
    {
        $response = $this->postJson(route('api.helix.points'), [
            'clientUserId' => '12',
            'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 13)
        ]);

        $response->assertStatus(Response::HTTP_FORBIDDEN);
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->json('code'));
        $this->assertEquals(trans('helix::messages.points.forbidden'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));
    }

    /**
     * @test
     */
    public function it_will_return_a_503_when_an_unknown_error_occurs()
    {
        User::create($this->getDefaultUserData());

        $this->app['config']->set('helix.user.attributes.points', null);

        $response = $this->postJson(route('api.helix.points'), [
            'clientUserId' => '1',
            'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 1)
        ]);

        $response->assertStatus(Response::HTTP_SERVICE_UNAVAILABLE);
        $this->assertEquals(Response::HTTP_SERVICE_UNAVAILABLE, $response->json('code'));
        $this->assertEquals(trans('helix::messages.points.unavailable'), $response->json('message'));
        $this->assertEquals(HelixResponse::STATUS_FAILURE, $response->json('status'));
    }

    /**
     * @test
     */
    public function it_will_return_a_503_when_the_user_model_is_not_configured()
    {
        User::create($this->getDefaultUserData());

        $this->app['config']->set('helix.user.model', null);

        $response = $this->postJson(route('api.helix.points'), [
            'clientUserId' => '1',
            'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 1)
        ]);

        $response->assertStatus(Response::HTTP_SERVICE_UNAVAILABLE);
    }

    /**
     * @test
     */
    public function it_will_return_a_503_when_the_user_id_attribute_is_not_configured()
    {
        User::create($this->getDefaultUserData());

        $this->app['config']->set('helix.user.attributes.id', null);

        $response = $this->postJson(route('api.helix.points'), [
            'clientUserId' => '1',
            'accessToken' => app(Token::class)->encrypt(config('helix.token.catalog'), 1)
        ]);

        $response->assertStatus(Response::HTTP_SERVICE_UNAVAILABLE);
    }
}
