<?php

namespace Ignite\Vendor\Helix\Laravel;

use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Illuminate\Support\Collection;
use Illuminate\Support\Fluent;

class Response implements Arrayable, Jsonable
{
    const STATUS_SUCCESS = 'success';
    const STATUS_FAILURE = 'error';

    const KEY_SSO_URL = 'URL';
    const KEY_HELIX_ID = 'helixUserId';

    /**
     * @var array
     */
    protected $attributes;

    /**
     * @var Collection
     */
    protected $collection;

    /**
     * Response constructor.
     *
     * @param array $attributes
     */
    public function __construct($attributes = [])
    {
        $this->attributes = $attributes;
    }

    /**
     * The status of the request.
     *
     * @return string
     */
    public function status()
    {
        return $this->get('status', self::STATUS_FAILURE);
    }

    /**
     * Determine if the status of the request failed.
     *
     * @return string
     */
    public function message()
    {
        return $this->get('message');
    }

    /**
     * The data as a collection of fluent objects.
     *
     * @return Collection
     */
    public function data()
    {
        if (! $this->collection) {
            $this->collection = (new Collection($this->attributes['data'] ?? []))
                ->mapInto(Fluent::class);
        }

        return $this->collection;
    }

    /**
     * Determine if the status of the request passed.
     *
     * @return string
     */
    public function passed()
    {
        return strtolower($this->get('status')) === self::STATUS_SUCCESS;
    }

    /**
     * Determine if the status of the request failed.
     *
     * @return string
     */
    public function failed()
    {
        return strtolower($this->get('status')) === self::STATUS_FAILURE;
    }

    /**
     * Get the value of the given key.
     *
     * @param  string $key
     * @param  mixed $default
     * @return mixed
     */
    public function get($key, $default = null)
    {
        if ($this->doesntHave($key)) {
            return $default;
        }

        return $this->attributes[$key];
    }

    /**
     * Check if the response attributes array contains the given key.
     *
     * @param string $key
     *
     * @return bool
     */
    public function has($key)
    {
        return array_key_exists($key, $this->attributes);
    }

    /**
     * Check if the response attributes array contains the given key.
     *
     * @param string $key
     *
     * @return bool
     */
    public function hasValue($key)
    {
        if ($this->doesntHave($key)) {
            return false;
        }

        return ! empty($this->get($key));
    }

    /**
     * Check if the response attributes array contains the given key.
     *
     * @param string $key
     *
     * @return bool
     */
    public function doesntHave($key)
    {
        return ! $this->has($key);
    }

    /**
     * Check if the response attributes array contains the given key and that value is empty.
     *
     * @param string $key
     *
     * @return bool
     */
    public function doesntHaveValue($key)
    {
        if ($this->doesntHave($key)) {
            return true;
        }

        return empty($this->get($key));
    }

    /**
     * Cast the response to an array.
     *
     * @return array
     */
    public function toArray()
    {
        return $this->attributes;
    }

    /**
     * Cast the response to JSON.
     *
     * @param int $options
     *
     * @return string
     */
    public function toJson($options = 0)
    {
        return json_encode($this->toArray(), $options);
    }

    /**
     * Cast the response to a string.
     *
     * @return string
     */
    public function __toString()
    {
        return $this->toJson();
    }

    /**
     * Proxy unrecognized methods to the data collection.
     *
     * @param string $name
     * @param array $paramaters
     * @return mixed
     * @throws \Exception
     */
    public function __call($name, $paramaters)
    {
        if (! method_exists($this->data(), $name)) {
            throw new \Exception("Unknown method {$name}().");
        }
        return call_user_func_array([$this->data(), $name], $paramaters);
    }
}
