<?php

namespace Ignite\Vendor\Helix\Laravel\Tests\Unit\Listeners;

use Ignite\Vendor\Helix\Laravel\Contracts\DecoratableSsoClientInterface;
use Ignite\Vendor\Helix\Laravel\Listeners\SyncOnRegister;
use Ignite\Vendor\Helix\Laravel\Response;
use Ignite\Vendor\Helix\Laravel\Sso\Client;
use Ignite\Vendor\Helix\Laravel\Sso\Payload;
use Ignite\Vendor\Helix\Laravel\Tests\Stubs\User;
use Ignite\Vendor\Helix\Laravel\Tests\TestCase;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Arr;
use Mockery;

class SyncOnRegisterTest extends TestCase
{
    /**
     * @test
     */
    public function it_will_sign_in_and_store_the_login_url_in_the_session()
    {
        $sessionKey = config('helix.sso.session.key');
        $response = $this->getResponseData([
            'URL' => $loginUrl = "http://52.60.111.154/brightspot/login/D5F9350EF38FC919E684023E68670584F951E3216391E53F28CDB64274D1C1EC"
        ]);
        $payload = new Payload($this->getPayloadData());

        request()->setLaravelSession(session());

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signIn')->once()->andReturn(new Response($response));

        $user = $this->getMockUser($payload);
        $user->shouldReceive('update')->with(Mockery::on(function ($argument) use ($response) {
            $key = config('helix.user.attributes.helix_user_id');
            return isset($argument[$key]) && $argument[$key] == 690901;
        }))->once();

        $listener = new SyncOnRegister($client);
        $listener->handle(new Registered($user));

        $this->assertEquals($loginUrl, session($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_error_when_you_have_not_configured_the_helix_user_id_attribute()
    {
        $this->app['config']->set('helix.user.attributes.helix_user_id', '');
        $sessionKey = config('helix.sso.session.key');
        $response = $this->getResponseData([
            'URL' => $loginUrl = "http://52.60.111.154/brightspot/login/D5F9350EF38FC919E684023E68670584F951E3216391E53F28CDB64274D1C1EC"
        ]);
        $payload = new Payload($this->getPayloadData());

        request()->setLaravelSession(session());

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signIn')->once()->andReturn(new Response($response));

        $user = $this->getMockUser($payload);
        $user->shouldReceive('jsonSerialize')->once()->andReturn($payload->toJson());

        $listener = new SyncOnRegister($client);
        $listener->handle(new Registered($user));

        $this->assertEquals(Client::SSO_UNKNOWN_ERROR, session($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_add_an_error_code_to_the_session_when_the_payload_validation_fails()
    {
        $sessionKey = config('helix.sso.session.key');
        $user = new User(['email' => 'bad email address']);
        request()->setLaravelSession(session());
        $client = app(DecoratableSsoClientInterface::class);

        $listener = new SyncOnRegister($client);
        $listener->handle(new Registered($user));

        $this->assertEquals(Client::SSO_INVALID_DATA, session($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_add_an_error_code_to_the_session_when_the_sso_response_contains_an_error()
    {
        $sessionKey = config('helix.sso.session.key');
        $response = $this->getResponseData([
            "code" => 500,
            "status" => "error",
            "message" => "Missing parameters: foo"
        ]);
        $payload = new Payload($this->getPayloadData());

        request()->setLaravelSession(session());

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signIn')->once()->andReturn(new Response($response));

        $user = $this->getMockUser($payload, $response);
        $user->shouldReceive('jsonSerialize')->once()->andReturn($payload->toJson());

        $listener = new SyncOnRegister($client);
        $listener->handle(new Registered($user));

        $this->assertEquals(Client::SSO_INVALID_RESPONSE, session($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_add_an_error_code_to_the_session_when_the_sso_login_url_is_malformed()
    {
        $sessionKey = config('helix.sso.session.key');
        $response = $this->getResponseData([
            'URL' => $loginUrl = "badly formed url"
        ]);
        $payload = new Payload($this->getPayloadData());

        request()->setLaravelSession(session());

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signIn')->once()->andThrow(new Response($response));

        $user = $this->getMockUser($payload);
        $user->shouldReceive('jsonSerialize')->once()->andReturn($payload->toJson());

        $listener = new SyncOnRegister($client);
        $listener->handle(new Registered($user));

        $this->assertEquals(Client::SSO_INVALID_LOGIN_URL, session($sessionKey));
    }

    /**
     * @test
     */
    public function it_will_add_an_error_code_to_the_session_when_a_general_error_occurs()
    {
        $sessionKey = config('helix.sso.session.key');
        $response = $this->getResponseData([
            'URL' => $loginUrl = "badly formed url"
        ]);
        $payload = new Payload($this->getPayloadData());

        request()->setLaravelSession(session());

        $client = Mockery::mock(DecoratableSsoClientInterface::class);
        $client->shouldReceive('signIn')->once()->andThrow(new \Exception('Generic error'));

        $user = $this->getMockUser($payload);
        $user->shouldReceive('jsonSerialize')->once()->andReturn($payload->toJson());

        $listener = new SyncOnRegister($client);
        $listener->handle(new Registered($user));

        $this->assertEquals(Client::SSO_UNKNOWN_ERROR, session($sessionKey));
    }

    /**
     * The mocked user model.
     *
     * @param Payload $payload
     *
     * @return User|Mockery\MockInterface
     */
    protected function getMockUser($payload)
    {
        $user = Mockery::mock(User::class);
        $user->shouldReceive('payload')->once()->andReturn($payload);

        return $user;
    }

    /**
     * The payload data.
     *
     * @param array $overrides
     *
     * @return array
     */
    private function getPayloadData($overrides = [])
    {
        return array_merge([
            //'helixUserId' => 690901,
            'clientUserId' => 1,
            'catalogId' => "A14E715F-CAF0-4749-B569-58059C9DD490",
            'accessToken' => "E7DFA4373E531A7AA23B1955B94E6526F4B33587FFE12A758DCB35E48736251B",
            'firstName' => 'John',
            'lastName' => 'Smith',
            'shipAddress1' => '102 Decker Ct',
            'shipAddress2' => 'Suite 150',
            'shipCity' => 'Irving',
            'shipState' => 'TX',
            'shipPostalCode' => '75065',
            'shipCountry' => 'US',
            'phoneNumber1' => '9726616031',
            'emailAddress' => 'it@brightspot.email'
        ], $overrides);
    }

    private function getResponseData($overrides = [])
    {
        $data = [
            'status' => "success",
        ];

        if (! isset($overrides['code']) || $overrides['code'] === 200) {
            $data = array_merge($data, [
                'code' => 200,
                'catalogId' => "A14E715F-CAF0-4749-B569-58059C9DD490",
                'clientUserId' => "1",
                'helixUserId' => 690901
            ]);
        }

        return array_merge($data, $overrides);
    }
}
